<?php

namespace Ignite\Claim\Models;

use Ignite\Claim\Repositories\ClaimRepository;

class Status
{
    /** @var ClaimRepository */
    protected $claimRepository;

    /**
     * Status constructor.
     *
     * @param ClaimRepository $claim_repository
     */
    public function __construct(ClaimRepository $claim_repository)
    {
        $this->claimRepository = $claim_repository;
    }

    /**
     * Update the status of the claim participant record.
     *
     * @param int   $id
     * @param array $data
     * @return bool
     */
    public function setStatus($id, $data)
    {
        $status = $data['status'];
        $params = [
            'class' => 'claim_participant',
            'action' => 'changeStatus',
            'claim_participant_id' => $id,
            'status' => $status,
        ];

        $expectsReason = ['declined', 'cancelled'];
        if (in_array($status, $expectsReason)) {
            $params['reason_declined'] = isset($data['reason']) ? $data['reason'] : '';
        }

        // ---------------------------------------
        // Update the Claim Status
        // NOTE:
        // Changing Claim Status this way causes other Business Logic
        // to be executed. Notably, Emails, and Promotion Cap enforcement.
        // ---------------------------------------
        return $this->claimRepository->put($params);
    }

    /**
     * Format the flash message.
     *
     * @param int    $total
     * @param int    $passed
     * @param int    $failed
     * @param string $status
     * @return $this
     */
    public function flashMessage($total, $passed, $failed, $status)
    {
        $claims = str_plural('Claim', $total);
        $status = ucfirst($status);

        if ($passed) {
            $message = sprintf(
                '%s/%s %s successfully updated to %s status.',
                $passed,
                $total,
                $claims,
                $status
            );
            flash()->success($message)->important();
        }

        if ($failed) {
            $message = sprintf(
                '%s/%s %s could not be updated to %s status.',
                $failed,
                $total,
                $claims,
                $status
            );
            flash()->error($message)->important();
        }

        return $this;
    }
}
