<?php

namespace Ignite\Claim\Models;

use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Ignite\Claim\Entities\Table;
use Ignite\Core\Models\Address;
use Ignite\Claim\Entities\Dynamic\DynamicModel;
use Ignite\Claim\Entities\ClaimParticipant;
use Ignite\Core\Repositories\ParticipantRepository;

class Form
{
    /**
     * Instance of the Participant repository class.
     * @var ParticipantRepository
     */
    public static $participantRepository;

    /**
     * A map of validation config.
     * @var array
     */
    public static $validationMap = [
        'integer'    => 'integer',
        'date'       => 'date',
        'datetime'   => 'date_format:m/d/Y H:i:s',
        'email'      => 'email',
        'float'      => 'numeric',
        'text'       => 'string',
        'time'       => 'date_format:H:i:s',
        'amount'     => 'amount',
        'dontrepeat' => 'dontrepeat:6'
    ];

    /**
     * Convert template options in the dynamic forms.
     *
     * @param  string $options
     * @return array
     * @throws \Exception
     */
    public static function convertOptions($options)
    {
        $options = trim($options);

        if (! preg_match('/^{([0-9a-zA-Z_]+)}$/', $options)) {
            return preg_split("/[|,\n]/", $options, -1, PREG_SPLIT_NO_EMPTY);
        }

        $variable = str_replace(['{', '}'], '', $options);
        $sources = config('sources');

        if (! isset($sources[$variable])) {
            throw new \Exception("Unknown data source `{$variable}`. Please update the sources.php config file.");
        }

        $source = app($sources[$variable]);

        return $source->toDropdown();
    }

    /**
     * Format a data.
     *
     * @todo   Laravel dates are automatically cast to Carbon instances which makes formatting dates simple, we should refactor so we don't need this method.
     * @param  string $value
     * @param  string $format
     * @return false|string
     */
    public static function format($value, $format)
    {
        if (! empty($value) && ! empty($format)) {
            switch ($format) {
                case 'date':
                    $value = date('m/d/Y', strtotime($value));
                    break;

                case 'datetime':
                    $value = date('m/d/Y h:i:sa', strtotime($value));
                    break;
            }
        }

        return $value;
    }

    /**
     * Get the column settings for the form.
     *
     * @todo Figure out the object represented by $tableSettings
     * @param  object $table
     * @param  string $columnName
     * @param  mixed  $default
     * @return mixed
     */
    public static function getColumnSettings($table, $columnName, $default = false)
    {
        if (isset($table->static_columns[$columnName])) {
            return $table->static_columns[$columnName];
        }

        if (isset($table->dynamic_columns[$columnName])) {
            return $table->dynamic_columns[$columnName];
        }

        return $default;
    }

    /**
     * Merge the form field data into our table column data.
     *
     * @todo Unit test this method.
     * @todo Too many parameters.
     *
     * @param  array $tableNames
     * @param  array|\Illuminate\Database\Eloquent\Model $formData
     * @param  array $formSettings
     * @param  bool  $review
     * @param  array $errors
     * @return array|bool
     * @throws \Exception
     */
    public static function mergeFormFieldData($tableNames, $formData, $formSettings, $review, &$errors)
    {
        // We always want to have an array of Table Names
        if (! is_array($tableNames)) {
            $tableNames = [$tableNames];
        }

        // Get the Table Column definitions
        $tableSettings = [];
        foreach ($tableNames as $tableName) {
            $key = DynamicModel::tableSettingKey($tableName);
            $tableModel = Table::findByKey($key);

            if (! $tableModel) {
                $errors[] = sprintf(
                    "Missing Table settings for Table = '%s', Key = '%s'\n",
                    $tableName,
                    $key
                );
                return false;
            }

            $tableSettings[$tableName] = $tableModel;
        }

        // Merge the Form Data with the Form Field params
        $fields = [];
        foreach ($formSettings->columns as $table => $columns) {
            if (empty($tableSettings[$table])) {
                continue;
            }

            foreach ($columns as $columnName => $params) {
                // 20170524 EWB fix bug with Form Edit page saving "--- select ---" when none selected
                $params['validateType'] = (strncmp($params['validateType'], '---', 3) != 0) ? $params['validateType'] : '';
                $params['value'] = (isset($formData[$columnName])) ? $formData[$columnName] : '';

                $tableColumn = self::getColumnSettings($tableSettings[$table], $columnName);
                if (empty($tableColumn)) {
                    $errors[] = sprintf("Unknown column = '%s' for Table = '%s'.\n", $columnName, $table);
                }

                if (! empty($tableColumn['options'])) {
                    $params['options'] = self::convertOptions($tableColumn['options']);
                }

                if ($review && ! in_array($params['type'], ['password', 'checkbox'])) {
                    $params['type'] = 'review';
                }

                $params['tableFormat'] = isset($tableColumn['format']) ? (int) $tableColumn['format'] : '';

                $params['sensitive'] = isset($tableColumn['sensitive']) ? (int) $tableColumn['sensitive'] : 0;

                $fields[$columnName] = (array) $params;
            }
        }

        // --------------------------------
        // Sort the Fields into the order specified
        // --------------------------------
        uasort($fields, function ($a, $b) {
            if ($a['order'] == $b['order']) {
                return 0;
            }
            return ($a['order'] > $b['order']) ? 1 : -1;
        });

        return $fields;
    }

    /**
     * Alter a rule if it matches a certain type.
     *
     * @todo   Ed is passing in the rule, acting upon it and return it,
     * @todo   looks like a utility method, find out what it's doing.
     * @param  string $rule
     * @param  array  $params
     * @return mixed|string
     */
    public static function getRule($rule, $params)
    {
        // ---------------------------------------
        // Handle case with existing Table params missing validateType
        // In this case the Validation will be in the params->validate string
        // ---------------------------------------
        if (isset($params['validateType'])) {
            $type = $params['validateType'];
        } else {
            $type = 'custom';
        }

        // ---------------------------------------
        // Handle case with validateType having the "--- select ---" in it
        // ---------------------------------------
        // SMELL: This is not something that should exist.
        // TODO: Remove the --- select --- label from the code base so that we can remove this next line
        if (strncmp($type, '---', 3) == 0) {
            $type = '';
        }

        if (! empty($type)) {
            switch ($type) {
                case 'custom':
                    if (! empty($params['validate'])) {
                        $rule = $params['validate'];
                    }
                    break;

                case 'none':
                    $rule = '';
                    break;

                case 'password':
                    // Need to come up with some rules for this
                    $rule = '';
                    break;

                default:
                    if (! empty(self::$validationMap[$type])) {
                        $rule = self::$validationMap[$type];
                    }
                    break;
            }
        }

        return $rule;
    }

    /**
     * Validate the form data.
     *
     * @param  string $tableName
     * @param  array  $formData
     * @param  object $formSettings
     * @return array
     */
    public static function validateFormData($tableName, $formData, $formSettings)
    {
        $validator = static::buildValidatorInstance($tableName, $formData, $formSettings);

        if ($validator->passes()) {
            return [];
        }

        return $validator->errors()->all();
    }

    public static function buildValidatorInstance($tableName, $formData, $formSettings)
    {
        $key = DynamicModel::tableSettingKey($tableName);
        $tableModel = Table::findByKey($key);

        if (! $tableModel) {
            $errors[] = sprintf("Missing Table settings for Table = '%s', Key = '%s'\n", $tableName, $key);
            return $errors;
        }

        // Loop through the fields and build the validation rules
        $rules = [];
        $attributeNames = [];

        foreach ($formSettings->columns as $table => $columns) {
            if ($table != $tableName) {
                continue;
            }

            foreach ($columns as $columnName => $formColumn) {
                $tableColumn = self::getColumnSettings($tableModel, $columnName, []);

                if (empty($tableColumn)) {
                    $errors[] = sprintf("Unknown column = '%s' for Table = '%s'.\n", $columnName, $tableName);
                }

                $rule = self::getRule('', $tableColumn);

                // Note:  Form Column Parameters override Table Column Parameters
                $rule = self::getRule($rule, $formColumn);

                if (! empty($rule)) {
                    if (str_contains($rule, 'unique')) {
                        $rule = static::replaceUniqueRule($table, $formData, $rule);
                    }
                    $rules[$columnName] = $rule;
                    $attributeNames[$columnName] = '"' . $formColumn['displayName'] . '"';
                }
            }
        }

        /** @var \Illuminate\Validation\Validator $validator */
        $validator = Validator::make($formData, $rules);
        $validator->setAttributeNames($attributeNames);

        return $validator;
    }

    /**
     * Find the primary key of the record type being validated.
     *
     * @param  string $table
     * @param  array  $formData
     * @return bool|string
     */
    public static function findKeyInData($table, $formData)
    {
        switch ($table) {
            case 'participant':
                $key = 'user_id';
                break;
            case 'claim':
            case 'claim_participant':
            case 'claim_lineitem':
                $key = 'id';
                break;
            default:
                $key = false;
        }

        if (array_key_exists($key, $formData)) {
            return $key;
        }

        return false;
    }

    /**
     * Replace the unique rule if we have a primary key in the formData.
     *
     * @param  string $table
     * @param  array  $formData
     * @param  string $rule
     * @return array
     */
    private static function replaceUniqueRule($table, $formData, $rule)
    {
        if ($key = static::findKeyInData($table, $formData)) {
            $ruleParts = explode('|', $rule);
            $rule = [];
            foreach ($ruleParts as $rulePart) {
                $ruleSpec = explode(':', $rulePart);
                $ruleName = $ruleSpec[0] ?? '';
                $ruleDetails = $ruleSpec[1] ?? '';

                if ('unique' === $ruleName) {
                    $ruleMeta = explode(',', $ruleDetails);
                    $table = $ruleMeta[0] ?? '';
                    $field = $ruleMeta[1] ?? '';
                    if ($table && $field) {
                        $rule[] = Rule::unique($table, $field)->ignore($formData[$key], $key);
                    }
                } else {
                    $rule[] = $ruleName . ':' . $ruleDetails;
                }
            }
        }

        return $rule;
    }
}
