<?php

namespace Ignite\Claim\Http\Controllers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Response;
use Ignite\Claim\Entities\File;
use Ignite\Claim\Entities\Exception;
use Ignite\Core\Http\Controllers\Controller;

class FileController extends Controller
{
    /**
     * Store a File.
     *
     * @return Response
     */
    public function store()
    {
        //$data = request()->all();

        try {
            // 20170726 EWB Had to disable this, as Laravel was throwing PostSize errors on all Uploads
            /*
            $validator = Validator::make($request->all(), [
                'related_type'      => 'required',
                'related_name'      => 'required',
                'related_id'        => 'required|integer',
                'file'              => 'required|file',
            ]);

            if ( $validator->fails() ) {
                $message = implode("\n", $validator->errors()->all());
                throw new Exception($message);
            }
            */

            $file = request()->file('file');

            list($fileName, $storagePath, $storeFileName) = $this->getUploadInfo($file);

            $fullStorePath = $file->storeAs($storagePath, $storeFileName, [
                'disk' => config('Claim.disk'),
                //'visibility' => 'public'
            ]);

            if (empty($fullStorePath)) {
                throw new Exception('Missing path to file.');
            }

            $response = $this->buildResponseObject(true, compact('fileName', 'storeFileName'));
        } catch (Exception $e) {
            $response = $this->buildResponseObject(false, false, $e->getMessage());
        }

        return response()->json($response, ($response->result) ? 200 : 422);
    }

    /**
     * Remove the File record and File from storage.
     *
     * @return void
     */
    public function delete()
    {
        //
    }

    /**
     * Provides File from storage as a download using an Obfuscated File Id
     * This is an Alias from get()
     *
     * @param  string $filehash
     * @return Response
     * @throws \Exception
     */
    public function show($filehash)
    {
        try {
            $file = File::getObfuscated($filehash);

            if (empty($filepath = setting($file->location_path_setting))) {
                throw new \Exception("No file path for Path Setting '%s'", $file->location_path_setting);
            }

            $name = $file->file_name;
            $path = "$filepath/$name";
            $stream = $this->getFilesystem()->readStream($path);
            $headers = [
                "Content-Type" => $this->getFilesystem()->getMimetype($path),
                "Content-Length" => $this->getFilesystem()->getSize($path),
                "Content-disposition" => "attachment; filename=\"$name\"",
            ];

            return response()->stream(function () use($stream) {
                fpassthru($stream);
            }, 200, $headers);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode()
            ], 404);
        }
    }

    /**
     * Build a response object for our Javascript.
     *
     * @param  bool  $result
     * @param  bool  $data
     * @param  array $messages
     * @return \stdClass
     */
    private function buildResponseObject($result = false, $data = false, $messages = [])
    {
        $response = new \stdClass;
        $response->result = $result;
        $response->data = $data;
        $response->messages = $messages;

        return $response;
    }

    /**
     * Get the file system driver.
     *
     * @return \League\Flysystem\Filesystem
     */
    private function getFilesystem()
    {
        return Storage::disk(config('Claim.disk'))->getDriver();
    }

    /**
     * Sanitize a file name.
     *
     * @param  string $fileName
     * @return false|string
     */
    private function sanitizeFileName($fileName)
    {
        // Found at: https://stackoverflow.com/questions/2021624/string-sanitizer-for-filename
        $fileName = mb_ereg_replace("([^\w\s\d\-_~,;\[\]\(\).])", '', $fileName);
        $fileName = mb_ereg_replace("([\.]{2,})", '', $fileName);

        return $fileName;
    }

    /**
     * Get the path to store the uploaded file.
     *
     * @param  string $relatedType
     * @param  string $relatedName
     * @return string
     */
    private function getStoragePath($relatedType, $relatedName)
    {
        $storagePath = 'files/' . $relatedType;

        if ($relatedType !== 'temp') {
            return $storagePath . '/' . $relatedName;
        }

        return $storagePath;
    }

    /**
     * Get the information from the request.
     *
     * @param  \Illuminate\Http\UploadedFile $file
     * @return array
     */
    private function getUploadInfo($file)
    {
        $relatedType = request()->get('related_type');
        $relatedName = request()->get('related_name');
        $relatedId = request()->get('related_id');

        $fileName = $file->getClientOriginalName();
        $cleanFileName = $this->sanitizeFileName($fileName);
        $storeFileName = $relatedName . '_' . $relatedId . '_' . $cleanFileName;
        $storagePath = $this->getStoragePath($relatedType, $relatedName);

        return [$fileName, $storagePath, $storeFileName];
    }
}
