<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use App;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Ignite\Claim\Entities\Offer;
use Ignite\Claim\Entities\OfferCampaign;
use Ignite\Claim\Entities\OfferPromotion;
use Ignite\Claim\Entities\OfferPromotionCap;
use Ignite\Claim\Entities\OfferPromotionGroup;
use Ignite\Claim\Entities\OfferUser;
use Ignite\Core\Repositories\ParticipantRepository;
use Ignite\Claim\Entities\Rule;

class OfferController extends Controller
{

    private $participantRepository;

    /**
     * Create a new controller instance.
     *
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(ParticipantRepository $participantRepository)
    {
        $this->participantRepository = $participantRepository;
    }

    /**
     * Display a listing of the resource.
	 *
     * @return Response
     */
    public function index()
    {
        return $this->showCampaigns();
    }

    /**
     * Display a listing of Campaigns
     * @return Response
     */
    public function showCampaigns()
    {
    	// TODO: Put this in the OfferRepository
        $campaigns = OfferCampaign::with(['hasPromotions', 'promotions'])
			->orderBy('name')
			->get()
			->toArray();

        return view('Claim::admin.offers.campaigns', compact('campaigns'));
    }

    /**
     * Display a listing of Campaigns
     * @return Response
     */
    public function showPromotions($campaignId = 1)
    {
        $data = request()->all();

        $campaigns = OfferCampaign::orderBy('name')->get()->toArray();
        $promotionGroups = OfferPromotionGroup::orderBy('name')->get()->toArray();
        $promotions = OfferPromotion::with(['hasOffers', 'offers'])
            ->where('campaign_id', '=', $campaignId)
            ->orderBy('name')
			->get()
			->toArray();

        $pageData = [
            'fiscalMonths'      => OfferPromotionCap::$fiscalMonths,
            'maxPeriods'        => OfferPromotionCap::$maxPeriods,
            'campaigns'         => $campaigns,
            'campaignId'        => $campaignId,
            'promotionGroups'   => $promotionGroups,
            'promotionGroupId'  => 1,
            'promotions'        => $promotions,
		];

        return view('Claim::admin.offers.promotions', $pageData);
    }

    /**
     * Display a listing of Promotion Groups.
	 *
     * @return Response
     */
    public function showPromotionGroups()
    {
        $promotionGroups = OfferPromotionGroup::with(['hasPromotions', 'promotions'])
			->orderBy('name')
			->get()
			->toArray();

        return view('Claim::admin.offers.promotionGroups', compact('promotionGroups'));
    }

    /**
     * Display a listing of Offers
     * @return Response
     */
    public function showOffers($promotionId = 1, $campaignId = 0)
    {
        $data = request()->all();

        $errors = [];

        // ---------------------------
        // Get All Campaigns and Promotions
        // ---------------------------
        $campaigns      = OfferCampaign::orderBy('name')->get()->toArray();
        $promotions     = OfferPromotion::orderBy('name')->get()->toArray();

        // ---------------------------
        // Get CampaignId from Promotion
        // ---------------------------
        if ( $campaignId == 0 ) {
            foreach($promotions as $promotion) {
                if ( $promotion['id'] == $promotionId ) {
                    $campaignId = $promotion['campaign_id'];
                    break;
                }
            }
        }

        if ( $campaignId == 0 ) {
            $errors[] = sprintf("Unable to determine Campaign for Promotion Id = %s.", $promotionId);
        }

        // ---------------------------
        // Remove Promotions not on this Campaign
        // ---------------------------
        foreach($promotions as $index => $promotion) {
            if ($promotion['campaign_id'] != $campaignId) {
                unset($promotions[$index]);
            }
        }

        if (count($promotions) == 0) {
            $errors[] = sprintf("No Promotions created for Campaign Id = %s.", $campaignId);
        }

        // -------------------------------------
        // Get List of Claims for Type requested for current User
        // -------------------------------------
        $offerSvc = app()->make(\Ignite\Claim\Repositories\OfferRepository::class);
        $params = [
        	'class'        => 'claim_offer',
			'action'       => 'getList',
			'mode'         => 'array',
            'with'         => ['values'],
			'promotion_id' => $promotionId,
			// 'status'       => '1'
		];
        $response = $offerSvc->get($params);
        $offers = $response->data;

        // Build lists for various selections
        $participantTypes = $this->participantRepository->getTypes();

        $pageData = [
            'valueTypeOptions' => Rule::$valueTypeOptions,
            'participantTypes' => $participantTypes,
            'campaigns'        => $campaigns,
            'campaignId'       => $campaignId,
            'promotions'       => $promotions,
            'promotionId'      => $promotionId,
            'offers'           => $offers,
		];

        return view('Claim::admin.offers.offers', $pageData)->with('errors', $errors);
    }

    public function showUserPromotionGroups()
    {
        $users = OfferUser::with('promotionGroups')
            ->orderBy('last', 'asc')
            ->orderBy('first', 'asc')
            ->get()
            ->toArray();

        return view('Claim::admin.offers.userPromotionGroup', compact('users'));
    }
}
