<?php

namespace Ignite\Claim\Entities;

use Illuminate\Database\Eloquent\SoftDeletes;
use Ignite\Claim\Entities\Dynamic\DynamicModel;
use Ignite\Claim\Traits\ValidatesFields;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

class Claim extends DynamicModel implements AuditableContract
{
    use Auditable, ValidatesFields, SoftDeletes;

    const TABLE_KEY = 'table.columns.claim';

    /** @var string */
    protected $table = 'claim';

    /** @var string */
    protected $primaryKey = 'id';

    /**
     * Should the timestamps be audited?
     *
     * @var bool
     */
    protected $auditTimestamps = false;

    /**
     * Custom Audit Driver
     *
     * @var \Ignite\Claim\Models\Audit\ClaimDriver
     */
    protected $auditDriver = \Ignite\Claim\Models\Audit\ClaimDriver::class;

    /**
     * Attributes to exclude from the Audit.
     *
     * @var array
     */
    protected $auditExclude = [
        'id',
        'updated_at',
        'created_at'
    ];

    /** @var array */
    public static $fields = [
        'id'                     => ['type' => 'integer'],
        'offer_promotion_id'     => ['type' => 'integer'],
        'parent_claim_id'        => ['type' => 'integer'],
        'claim_type'             => ['type' => 'string'],
        'user_id'                => ['type' => 'integer'],
        'participant_type'       => ['type' => 'string'],
        'date_submitted'         => ['type' => 'date'],
        'customer_name'          => ['type' => 'string', 'default' => '', 'validate' => 'required'],
        'customer_contact_first' => ['type' => 'string', 'default' => '', 'validate' => 'required'],
        'customer_contact_last'  => ['type' => 'string', 'default' => '', 'validate' => 'required'],
        'customer_contact_title' => ['type' => 'string', 'default' => ''],
        'customer_contact_email' => ['type' => 'string', 'default' => '', 'validate' => 'required|email|max:255'],
        'customer_contact_phone' => ['type' => 'string', 'default' => '', 'validate' => 'required|max:32'],
        'customer_country'       => ['type' => 'string'],
        'customer_address_1'     => ['type' => 'string'],
        'customer_address_2'     => ['type' => 'string'],
        'customer_city'          => ['type' => 'string'],
        'customer_state'         => ['type' => 'string'],
        'customer_postal'        => ['type' => 'string'],
        'deal_registration_date' => ['type' => 'date'],
        'distributor_po_number'  => ['type' => 'string'],
        'prm_number'             => ['type' => 'string'],
        'product_type'           => ['type' => 'string'],
        'activity_date'          => ['type' => 'date'],
        'sale_value'             => ['type' => 'double'],
        'estimated_revenue'      => ['type' => 'double'],
        'revenue_sold'           => ['type' => 'double'],
        'internal_notes'         => ['type' => 'string'],
        'archived'               => ['type' => 'integer', 'default' => '0'],
    ];

    /**
     * The relationship to the participant.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo('Ignite\Core\Entities\Participant', 'user_id', 'user_id');
    }

    /**
     * The relationship to the user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo('Ignite\Core\Entities\User', 'user_id', 'user_id');
    }

    /**
     * The relationship to the claim line items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lineitems()
    {
        return $this->hasMany('Ignite\Claim\Entities\ClaimLineitem', 'claim_id', 'id');
    }

    /**
     * The relationship to the claim participants.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function claim_participants()
    {
        return $this->hasMany('Ignite\Claim\Entities\ClaimParticipant', 'claim_id', 'id');
    }

    /**
     * The relationship to the documents.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function documents()
    {
        return $this->hasMany('Ignite\Claim\Entities\File', 'related_id', 'id')
            ->where('claim_file.related_type', '=', 'table')
            ->where('claim_file.related_name', '=', 'claim');
    }

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return 'customer_name';
    }
}
