<?php

namespace Ignite\Claim\Http\Controllers\Admin;

use Ignite\Claim\Entities\Form;
use Ignite\Claim\Entities\Table;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Claim\Entities\Dynamic\DynamicModel;
use Ignite\Claim\Entities\Setting;

class FormsController extends Controller
{
    public static $modules = [
        'claim'         => 'claim',
    ];

    public static $formTypes = [
        'dialog'        => 'dialog',
        'page'          => 'page',
        'partial'       => 'partial',
    ];

    public static $formColumnTypes = [
        'checkbox'        => 'checkbox',
        'date'            => 'date',
        'datetime'        => 'datetime',
        'file'            => 'file',
        'hidden'          => 'hidden',
        'password'        => 'password',
        'radio'           => 'radio',
        'review'          => 'review',
        'select'          => 'select',
        'selecttext'      => 'selecttext',
        'multiselect'     => 'multiselect',
        'multiselecttext' => 'multiselecttext',
        'text'            => 'text',
        'textarea'        => 'textarea',
    ];

    /**
     * List all of the available forms.
     *
     * @return \Illuminate\Contracts\View\Factory
     */
    public function index()
    {
        $forms = Form::all();

        return view('Claim::admin.settings.form.index', compact('forms'));
    }

    /**
     * Create a new form using a key.
     *
     * @return \Illuminate\Contracts\View\Factory
     */
    public function create()
    {
        $pageData = [
            'modules' => self::$modules,
            'formTypes' => self::$formTypes,
            'formTables' => DynamicModel::$tables,
        ];

        return view('Claim::admin.settings.form.create', $pageData);
    }

    public function edit($key = '')
    {
        try {
            $form = $this->formatKey($key);
        } catch (\Exception $e) {
            flash()->error($e->getMessage())->important();
            return back();
        }

        $tables = DynamicModel::$tables;
        $tableColumns = $this->getTableColumnsDefinition($tables);
        $tableDiff = array_diff(array_keys($tables), array_keys($tableColumns));

        /*if (! empty($tableDiff)) {
            $message = sprintf('Unable to find settings for table(s): %s', implode(', ', $tableDiff));
            flash()->error($message);
            return back();
        }*/

        // Get Form Configuration
        $formModel = Form::findByKey($key);

        if (! $formModel) {
            $formModel = Form::create([
                'key' => $key,
                'module' => $form['module'],
                'type' => $form['type'],
                'name' => $form['name'],
                'columns' => []
            ]);
        }

        return view('Claim::admin.settings.form.edit', [
            'tableColumns' => $tableColumns,
            'formColumns' => $formModel,
            'formColumnTypes' => self::$formColumnTypes,
            'validateTypes' => DynamicModel::getValidateTypes(),
            'formFields' => $this->getFormFieldsDefinition(),
        ]);
    }

    /**
     * Delete a form.
     *
     * @param  string $formKey
     * @return \Illuminate\Http\RedirectResponse
     */
    public function delete($formKey)
    {
        $form = Form::findByKey($formKey);

        if (empty($form)) {
            flash()->error(sprintf("No Form found for Key = '%s'.", $formKey))->important();
            return back();
        }

        try {
            $form->delete();
        } catch (\Exception $e) {
            flash()->error(sprintf("Unable to delete form for key: '%s'.", $formKey))->important();
            return back();
        }

        flash()->success(sprintf("Successfully deleted the form with key: '%s' .", $formKey))->important();

        return back();
    }

    /**
     * Get the form fields to display when creating a form.
     *
     * @return array
     */
    private function getFormFieldsDefinition()
    {
        $fields = [
            'order'             => [
                'formName'      => 'Display Order',
                'dbType'        => 'integer',
                'required'      => false,
                'formType'      => 'text',
                'default'       => '',
                'editAllow'     => true,
            ],
            'table'           => [
                'formName'      => 'Table Name',
                'dbType'        => 'string',
                'required'      => true,
                'formType'      => 'select',
                'default'       => '',
                'editAllow'     => false,
            ],
            'name'           => [
                'formName'      => 'Column Name',
                'dbType'        => 'string',
                'required'      => true,
                'formType'      => 'select',
                'default'       => '',
                'editAllow'     => false,
            ],
            'displayName'    => [
                'formName'      => 'Display Name',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'text',
                'default'       => '',
                'editAllow'     => true,
            ],
            'type'           => [
                'formName'      => 'Form Field Type',
                'dbType'        => 'string',
                'required'      => true,
                'formType'      => 'select',
                'default'       => '',
                'options'       => [],
                'editAllow'     => true,
            ],
            'required'       => [
                'formName'      => 'Required',
                'dbType'        => 'bool',
                'required'      => false,
                'formType'      => 'checkbox',
                'default'       => true,
                'editAllow'     => true,
            ],
            'validateType'       => [
                'formName'      => 'Validation Type',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'select',
                'default'       => '',
                'options'       => [],
                'editAllow'     => true,
            ],
            'validate'       => [
                'formName'      => 'Laravel Validation String',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'text',
                'default'       => '',
                'editAllow'     => true,
            ],
            'format'         => [
                'formName'      => 'Format',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'text',
                'default'       => '',
                'editAllow'     => true,
            ],
            'helpText'         => [
                'formName'      => 'Help Text',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'textarea',
                'default'       => '',
                'editAllow'     => true,
            ],
            'default'         => [
                'formName'      => 'Default Value',
                'dbType'        => 'string',
                'required'      => false,
                'formType'      => 'text',
                'default'       => '',
                'editAllow'     => true,
            ],
        ];

        $fields['type']['options'] = self::$formColumnTypes;
        $fields['validate']['options'] = DynamicModel::getValidateTypes();

        return $fields;
    }

    /**
     *
     * @param  string $key
     * @return array
     * @throws \Exception
     */
    private function formatKey($key)
    {
        $key = trim($key);

        if (empty($key)) {
            throw new \Exception('The form key was empty.');
        }

        $parts = explode('.', $key);

        $form = [];
        $form['module'] = (! empty($parts[1])) ? trim($parts[1]) : '';
        $form['type'] = (! empty($parts[2])) ? trim($parts[2]) : '';
        $form['name'] = (! empty($parts[3])) ? trim($parts[3]) : '';

        foreach ($form as $name => $value) {
            if (empty($form[$name])) {
                $errors[] = $name;
            }
        }

        if (! empty($errors)) {
            throw new \Exception(sprintf(
                'The form key is missing required portions: "%s".',
                implode(', ', $errors)
            ));
        }

        return $form;
    }

    private function getTableColumnsDefinition($tables)
    {
        $tableColumns = [];

        foreach ($tables as $tableName => $params) {
            $table = $this->getTableByTableName($tableName);

            if (empty($table)) {
                continue;
            }

            $tableColumns[$tableName] = $table->getMergedColumns();
        }

        return $tableColumns;
    }

    /**
     * Get the table record by table name.
     *
     * @param  string $table
     * @return Table
     */
    private function getTableByTableName($table)
    {
        $key = DynamicModel::tableSettingKey($table);

        return Table::findByKey($key);
    }
}
