<?php

namespace Ignite\Claim\Repositories;

use Ignite\Claim\Entities\Exception;
use Ignite\Claim\Entities\Setting;
use Ignite\Claim\Entities\Translation;
use Ignite\Claim\Repositories\DataRepository;
use Ignite\Claim\Traits\ServiceProviderCommon;

class CoreRepository
{
    use ServiceProviderCommon;

    protected $classes          = [];
    protected $nickNames        = [];

    protected $processed        = false;
    protected $lastParams       = false;
    protected $response         = false;

    public function __construct($classes)
    {
        if (! is_array($classes)) {
            $classes = [$classes];
        }

        foreach ($classes as $classInfo) {
            $classPath = $classInfo['path'];
            $className = basename(str_replace('\\', '/', $classPath));
            $nickName = (! empty($classInfo['nickName']))
                ? $classInfo['nickName']
                : $this->getClassNickName($className);

            $class = [];
            $class['className']       = $className;
            $class['nickName']        = $nickName;
            $class['classPath']       = $classPath;
            $class['filters']         = (! empty($classInfo['filters'])) ? $classInfo['filters'] : [];
            $class['defaultTypesKey'] = (! empty($classInfo['defaultTypesKey'])) ? $classInfo['defaultTypesKey'] : [];
            $class['dataProvider']    = new DataRepository($classPath, $class['filters']);

            // Check for Column Mis-matches on this
            /*
            $errors = [];
            $class['dbColumnsChanged']  = $class['dataProvider']->checkDbColumnsChanged($errors);
            $class['dbColumnsErrors']   = $errors;

            if ( $class['dbColumnsChanged'] ) {
                $message = implode("\n", $errors);
                throw new Exception($message);
            }
            */

            // Save all the Table info both by Class and Nickname (table name)
            $this->classes[$nickName]       = $class;
            $this->nickNames[$className]    = $nickName;
        }
    }

    public function getClassNickName($className)
    {
        $nickName = '';
        $length = strlen($className);

        for ($ii = 0; $ii < $length; ++$ii) {
            $oneChar = $className[$ii];
            if ($oneChar < 'a' && $ii > 0) {
                $nickName .= '_' . strtolower($oneChar);
            } else {
                $nickName .= strtolower($oneChar);
            }
        }

        return $nickName;
    }

    public function getClassFilters($class)
    {
        $filters =  (! empty($this->classes[$class])) ? $this->classes[$class]['filters'] : [];

        return $filters;
    }

    /*
     * This function creates an array of Filter Parameters that were in
     * the RequestParameters for the Class requested.
     * If params is supplied, only filters not already in the params variable are returned.
     * @param string $class
     * @param array $requestParams
     * @param array $params
     *
     * @return array
     */
    public function getFilterParams($class, $requestParams, $params = [])
    {
        $filters = [];

        $classFilters = $this->getClassFilters($class);

        foreach ($classFilters as $name => $filter) {
            if (! empty($requestParams[$name]) && empty($params[$name])) {
                $filters[$name] = $requestParams[$name];
            }
        }

        return $filters;
    }

    public function __get($name)
    {
        $value = false;

        switch ($name) {
            case 'errors':
                $value = $this->response->errors;
                break;

            case 'lastParams':
                $value = $this->lastParams;
                break;

            case 'lastResult':
                $value = $this->response->result;
                break;

            case 'lastResponse':
                $value = $this->response;
                break;
        }

        return $value;
    }

    public function getSvcData($className, $name)
    {
        $function = __FUNCTION__;

        $params = [];
        $params['class']    = $className;

        $this->initResponse($params);

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        $value = false;

        switch ($name) {
            case 'lastParams':
                $value = $dataProvider->lastParams;
                break;

            case 'lastResult':
                $value = $dataProvider->response->result;
                break;

            case 'errors':
                $value = $dataProvider->response->errors;
                break;

            case 'lastRepsonse':
                $value = $dataProvider->response;
                break;

            case 'class':
                $value = $dataProvider->svc_class;
                break;

            case 'classPath':
                $value = $dataProvider->svc_classPath;
                break;

            case 'dataFields':
                $value = $dataProvider->svc_dataFields;
                break;

            case 'filters':
                $value = $dataProvider->svc_filters;
                break;
        }

        return $value;
    }

    public function getLastResponse()
    {
        return $this->response;
    }

    private function getClass($function, $className)
    {
        if (! empty($this->classes[$className])) {
            return $this->classes[$className];
        }

        if (! empty($this->nickNames[$className])) {
            $nickName = $this->nickNames[$className];
            return $this->classes[$nickName];
        }

        $this->response->result   = false;
        $this->response->errors[] = sprintf(
            "%s: Class name '%s' not registered.",
            $function,
            $className
        );

        return false;
    }

    /**
     * @param $function
     * @param $className
     *
     * @return bool|DataRepository
     */
    private function getDataProvider($function, $className)
    {
        $class = $this->getClass($function, $className);

        if ($class) {
            return $class['dataProvider'];
        }

        return false;
    }

    protected function getTypeList($params)
    {
        $function = __FUNCTION__;

        // ----------------------------------
        // Get Wildcard Key for looking up Types in Settings
        // ----------------------------------
        $requireds = ['wildKey'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        // ----------------------------------
        // Get Types Names/Values from Settings
        // ----------------------------------
        $types = Setting::byKey($params['wildKey']);

        if (empty($types)) {
            $this->response->result = false;
            $this->response->errors[] = sprintf(
                "%s: '%s' missing from setting table.",
                $function,
                $params['wildKey']
            );

            return $this->response->result;
        }

        // ----------------------------------
        // Get Types Texts from Translations
        // ----------------------------------
        $typeKeyPath = str_replace('*', '', $params['wildKey']);
        $resultTypes = new \stdClass();

        foreach ($types as $key => $value) {
            $keyParts = explode('.', $key);
            $keyBase = end($keyParts);

            $xlateKey = $typeKeyPath . $keyBase;
            $translation = Translation::byKey($xlateKey);

            if ($translation) {
                $translationValue = $translation->value;
                $resultTypes->$translationValue = $translationValue;
            } else {
                $resultTypes->$value = "MISSING_XLATE for $xlateKey";
            }
        }

        // ----------------------------------
        // Convert the response->data into the correct format
        // ----------------------------------
        $mode = (! empty($params['mode'])) ? $params['mode'] : 'db';

        switch ($mode) {
            case 'array':
                $this->response->data = (array) $resultTypes;
                break;

            case 'json':
                $this->response->data = json_encode($resultTypes);
                break;

            case 'db':
            default:
                $this->response->data = $resultTypes;
                break;
        }

        return $this->response;
    }

    /**
     * @param $className
     *
     * @return bool
     */
    public function setDefaultValues($className)
    {
        $function = __FUNCTION__;

        $params = [];
        $params['class'] = $className;

        $this->initResponse($params);

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        return $dataProvider->setDefaultValues();
    }

    /**
     *
     *
     * @param array $data
     * @param array $fields
     * @return bool
     */
    public function data($data, $fields = [])
    {
        $function = __FUNCTION__;

        $params = [];
        $params['data']     = $data;
        $params['fields']   = $fields;

        $this->initResponse($params);

        $requireds = ['class'];
        if (! $this->checkRequireds($function, $data, $requireds)) {
            return false;
        }

        $className = $data['class'];

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        return $dataProvider->data($data, $fields);
    }

    /**
     * Cast the data to an array.
     *
     * @param  string $className
     * @return bool
     */
    public function toArray($className)
    {
        $function = __FUNCTION__;
        $params = [];
        $params['class'] = $className;
        $this->initResponse($params);

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        return $dataProvider->toArray();
    }

    /**
     * Validate the data for a class using the given fields.
     *
     * @param  string $className
     * @param  array  $data
     * @param  bool   $validateFields
     * @return bool|array
     */
    public function validate($className, $data, $validateFields = false)
    {
        $function = __FUNCTION__;
        $params = [];
        $params['class'] = $className;
        $params['data'] = $data;
        $params['validateFields'] = $validateFields;
        $this->initResponse($params);

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        return $dataProvider->validate($data, $validateFields);
    }

    /**
     * Check the required fields.
     *
     * @param  string $function
     * @param  array  $params
     * @param  array  $requireds
     * @return mixed
     */
    protected function checkRequireds($function, $params, $requireds)
    {
        $checkParams = self::isArray($params) ? $params[0] : $params;

        foreach ($requireds as $required) {
            if (! array_key_exists($required, $checkParams)) {
                $this->response->result = false;
                $this->response->errors[] = sprintf("%s: missing param [%s].", $function, $required);
            }
        }

        return $this->response->result;
    }

    public function get($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action', 'class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $action     = $params['action'];
        $className  = $params['class'];

        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return false;
        }

        switch ($action) {
            case 'find':
            case 'getList':
                $this->response = $dataProvider->get($params);
                break;

            case 'getTypeList':
                $this->response = $this->getTypeList($params);
                break;

            default:
                $this->response->result = false;
                $this->response->errors[] = sprintf(
                    "%s: Unknown action param '%s'.",
                    $function,
                    $action
                );
                break;
        }

        return $this->response;
    }

    public function create($params)
    {
        $function = __FUNCTION__;
        $this->initResponse($params);

        $requireds = ['class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $className = self::getClassFromParams($params);
        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return $this->response;
        }

        $this->response = $dataProvider->create($params);

        return $this->response;
    }

    public function update($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $className = self::getClassFromParams($params);
        $dataProvider = $this->getDataProvider($function, $className);

        if (! $dataProvider) {
            return $this->response;
        }

        $this->response = $dataProvider->update($params);

        return $this->response;
    }

    public function delete($params)
    {
        $function = __FUNCTION__;
        $this->initResponse($params);

        $requireds = ['class'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $className = self::getClassFromParams($params);
        $dataProvider = $this->getDataProvider($function, $className);
        if (! $dataProvider) {
            return $this->response;
        }

        $this->response = $dataProvider->delete($params);

        return $this->response;
    }

    public function sync($dataItems, $className, $foreignKeyName, $foreignKeyValue = false)
    {
        $function = __FUNCTION__;
        $this->initResponse($dataItems);
        $dataProvider = $this->getDataProvider($function, $className);

        if (! $dataProvider) {
            return $this->response;
        }

        $this->response = $dataProvider->sync($dataItems, $className, $foreignKeyName, $foreignKeyValue);

        return $this->response;
    }
}
