<?php

namespace Ignite\Claim\Entities\Dynamic;

use Illuminate\Database\Eloquent\Model;

abstract class DynamicModel extends Model
{
    /** @var bool */
    public $isDynamicModel = true;

    /** @var bool|\Illuminate\Config\Repository|mixed */
    private $connectionName = false;

    /** @var string */
    protected $table = false;

    /** @var bool|\Illuminate\Config\Repository|mixed */
    private $dbConfig = false;

    /** @var bool|mixed */
    private $databaseName = false;

    /** @var bool|string */
    private $tableName = false;

    /** @var bool|MariaDB */
    protected $IgniteDb = false;

    /** @var bool */
    protected $dbColumns = false;

    /** @var bool|DynamicColumns */
    public $dynamic = false;

    /** @var array */
    public static $tables = [
        'claim'             => ['class' => 'Ignite\Claim\Entities\Claim'],
        'claim_lineitem'    => ['class' => 'Ignite\Claim\Entities\ClaimLineitem'],
        'claim_participant' => ['class' => 'Ignite\Claim\Entities\ClaimParticipant'],
        'claim_offer'       => ['class' => 'Ignite\Claim\Entities\Offer'],
    ];

    /** @var array */
    public static $dbColumnTypes = [
        'bool'      => 'bool',
        'string'    => 'string',
        'int'       => 'int',
        'double'    => 'double',
        'date'      => 'date',
        'datetime'  => 'datetime',
    ];

    /** @var array */
    public static $validateTypes = [
        'none'          => 'none',
        'custom'        => 'custom',
        'date'          => 'date',
        'datetime'      => 'datetime',
        'email'         => 'email',
        'float'         => 'float',
        'integer'       => 'integer',
        // 'range'         => 'range',
        'text'          => 'text',
        'time'          => 'time',
        'password'      => 'password',
        'amount'        => 'amount',
        'emaildomain'   => 'emaildomain'
        // 'us_currency'   => 'us_currency',
    ];

    /**
     * DynamicModel constructor.
     *
     * @param array $attributes
     */
    public function __construct(array $attributes = [])
    {
        $this->connectionName = config('database.default', false);
        $this->dbConfig = config('database.connections.' . $this->connectionName, false);
        $this->databaseName = $this->dbConfig['database'];
        $this->tableName = $this->table;
        $this->IgniteDb = IgniteDB::create($this->dbConfig['driver']);

        unset($this->dbConfig['username']);
        unset($this->dbConfig['password']);

        // Default the Static Columns to the Database Schema Columns
        $this->setDbColumns();

        // Override the default Field Definitions in the Model Class.
        // This is done to provide both Static and Dynamic fields to all
        // all the functions that rely on the Field Definitions.
        $this->dynamic = new DynamicColumns($this->IgniteDb, $this->table, $this->databaseName);

        parent::__construct($attributes);
    }

    /**
     * Get the table setting key.
     *
     * @param string $tableName
     * @return string
     */
    public static function tableSettingKey($tableName = '')
    {
        $key = 'table.columns.' . $tableName;

        return $key;
    }

    /**
     * Get the table data fields.
     *
     * @return mixed
     */
    public function getDataFields()
    {
        // Make sure we have retrieved the Columns from the Settings before we can return them
        $this->setDataFields();

        return static::$fields;
    }

    /**
     * The validation types.
     *
     * @return array
     */
    public static function getValidateTypes()
    {
        return static::$validateTypes;
    }

    /**
     * The database column types.
     *
     * @return array
     */
    public static function getDbColumnTypes()
    {
        return static::$dbColumnTypes;
    }

    /**
     * The dynamic tables.
     *
     * @return array
     */
    public static function getTables()
    {
        return static::$tables;
    }

    /**
     * Set the table data fields.
     *
     * @return mixed
     */
    public function setDataFields()
    {
        if (! static::$fieldsUpdated) {
            $dataFields = array_merge(
                $this->dynamic->getDbColumns(),
                $this->dynamic->getDynamicColumns()
            );

            static::setFields($dataFields);
        }

        return static::$fields;
    }

    /**
     * Determine if the given column is dynamic.
     *
     * @param  string $column
     * @return bool
     */
    public function isDynamicColumn($column)
    {
        return $this->dynamic->isDynamicColumn($column);
    }

    /**
     * Determine if the current module has a dynamic field.
     *
     * @return int
     */
    public function hasDynamic()
    {
        return $this->dynamic->hasDynamic();
    }

    /**
     * The dynamic column name.
     *
     * @return bool|string
     */
    public function getDynamicColumnName()
    {
        return $this->dynamic->getDynamicColumnName();
    }

    /**
     * Get the local vars.
     *
     * @param string $name
     * @return bool|mixed|null
     */
    public function getLocal($name)
    {
        static $locals = [
            'connectionName',
            'dbConfig',
            'databaseName',
            'IgniteDb',
            'dynamic',
            'tableName',
        ];

        if (in_array($name, $locals)) {
            return $this->$name;
        }

        return false;
    }

    /**
     * This function is NOT redundant to the one in the DynamicColumns class
     * it is needed to handle First Time creation of new Table Settings.
     * It goes directly to the Database Schema for the Db Columns and
     * doesn't rely on the Settings Table. So it won't cause the constructor
     * to fail if no Settings exist for this Table.
     *
     * @param  bool $tableName
     * @param  bool $databaseName
     * @return bool|mixed
     */
    private function getDbColumns($tableName = false, $databaseName = false)
    {
        // Check if we are getting the Database Columns for the Inherited Class
        if ($tableName === false && $databaseName === false && $this->dbColumns !== false) {
            return $this->dbColumns;
        }

        // Get Database Name for the Inherited Class
        $databaseName = (! $databaseName) ? $this->databaseName : $databaseName;

        if ($databaseName == '') {
            return false;
        }

        // Get Table Name for the Inherited Class
        $tableName = (! $tableName) ? $this->table : $tableName;

        $this->dbColumns = $columns = $this->IgniteDb->getColumns($databaseName, $tableName);

        return $columns;
    }

    /**
     * Set the database columns.
     *
     * @return void
     */
    private function setDbColumns()
    {
        $this->dbColumns = $this->getDbColumns();
    }

    /**
     * Get the dynamic columns.
     *
     * @return array|bool|string
     */
    public function getDynamicColumns()
    {
        return $this->dynamic->getDynamicColumns();
    }

    /**
     * Magic getter.
     *
     * @param  string $name
     * @return mixed|null
     */
    public function __get($name)
    {
        // Make sure we have retrieved the Columns from the Settings before we check anything
        $this->setDataFields();

        // Check Static Columns first to avoid needing Dynamic Columns if not defined
        if (isset($this->dbColumns[$name])) {
            return $this->getAttribute($name);
        }

        // Check if it is a Known Dynamic Column
        if ($this->dynamic->isDynamicColumn($name)) {
            return $this->dynamic->getColumn($this, $name);
        }

        // Not a Known Static or Dynamic Column, fall back to Eloquent to handle
        return parent::__get($name);
    }

    /**
     * Magic setter.
     *
     * @param string $name
     * @param mixed $value
     * @return bool|void
     */
    public function __set($name, $value)
    {
        // Make sure we have retrieved the Columns from the Settings before we check anything
        $this->setDataFields();

        // Check Static Columns first to avoid needing Dynamic Columns if not defined
        if (isset($this->dbColumns[$name])) {
            $this->setAttribute($name, $value);
            return;
        }

        // Check if it is a Known Dynamic Column
        if ($this->dynamic->isDynamicColumn($name)) {
            $this->dynamic->setColumn($this, $name, $value);
            return;
        }

        // Not a Known Static or Dynamic Column, fall back to Eloquent to handle
        parent::__set($name, $value);
        return;
    }

    /**
     * Determine whether the db columns changed.
     *
     * @param  array $errors
     * @return bool
     */
    public function checkDbColumnsChanged(&$errors)
    {
        $tmpErrors = [];
        $dbColumns      = $this->dynamic->getDbColumns();
        $nativeColumns  = IgniteDB::getColumnsNative($this->table);

        foreach ($dbColumns as $name => $column) {
            if (! isset($nativeColumns[$name])) {
                $tmpErrors[] = sprintf(
                    "Settings Table %s Column '%s' missing on Database Columns.",
                    $this->tableName,
                    $name
                );
            }
        }

        foreach ($nativeColumns as $name => $column) {
            if (! isset($dbColumns[$name]) && $name != $this->dynamic->getDynamicColumnName()) {
                $tmpErrors[] = sprintf(
                    "Database Table %s Column '%s' missing on Settings Columns.",
                    $this->tableName,
                    $name
                );
            }
        }

        if (! empty($tmpErrors)) {
            $message = sprintf(
                "Database Table '%s' and Settings Columns out of sync:",
                $this->tableName
            );
            array_unshift($tmpErrors, $message);
            $errors = array_merge($errors, $tmpErrors);
            return true;
        }

        return false;
    }
}
