<?php

namespace Ignite\Claim\Traits;

use Ignite\Claim\Entities\Form;

trait SupportFormSchema
{
    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'idc', 'id'
    ];

    /**
     * The column types to exclude.
     * @var array
     */
    protected $excludedTypes = ['textarea'];

    /**
     * The column keys that should be always visible.
     * @var array

    protected $alwaysVisible = [];*/

    /**
     * The cache of form and table data.
     * @var array
     */
    protected $formColumns;

    /**
     * Decorate the report with Form schema data.
     *
     * @param  array $columns
     * @param  array $sensitive
     * @return array
     */
    protected function decorateFormSchema($columns, $sensitive)
    {
        $formColumns = $this->getFormColumns($this->forms, $sensitive);
        $columns = array_merge($columns, $formColumns);

        foreach (array_diff_key($columns, $formColumns) as $key => $column) {
            if (! in_array($key, $this->alwaysVisible)) {
                $columns[$key] = array_merge(['visible' => false], $column);
            }
        }

        return $columns;
    }

    /**
     * Fetch the form columns specified in the concrete class, if any.
     *
     * @param  array $formKeys
     * @param  array $sensitive
     * @return array
     */
    protected function getFormColumns($formKeys, $sensitive)
    {
        if (isset($this->formColumns)) {
            return $this->formColumns;
        }

        $fields = [];

        foreach (Form::whereIn('key', $formKeys)->pluck('columns') as $forms) {
            foreach ($forms as $table => $cols) {
                foreach ($cols as $col => $column) {
                    if ($this->isValid($column)) {
                        $fields[$col] = $this->decorateColumn($table, $column, $sensitive);
                    }
                }
            }
        }

        return $this->formColumns = $fields;
    }

    /**
     * Validate the column is not an excluded type or key.
     *
     * @param  array $column
     * @return bool
     */
    protected function isValid($column)
    {
        return $this->isNotExcludedType($column['type']) && $this->isNotExcludedKey($column['name']);
    }

    /**
     * Ensure the given type is not in the list of excluded types.
     *
     * @param  string $type
     * @return bool
     */
    protected function isNotExcludedType($type)
    {
        return ! in_array($type, $this->excludedTypes);
    }

    /**
     * Ensure the given key is not in the list of excluded keys.
     *
     * @param  string $key
     * @return bool
     */
    protected function isNotExcludedKey($key)
    {
        return ! in_array($key, $this->excludedKeys);
    }

    /**
     * Decorate the column date.
     *
     * @param  string $table
     * @param  array  $column
     * @param  array  $sensitive
     * @return array
     */
    protected function decorateColumn($table, $column, $sensitive)
    {
        $key = $table . '.' . $column['name'];

        return [
            'title' => $column['displayName'],
            'name' => $key,
            'orderable' => true,
            'exportable' => true,
            'searchable' => true,
            'visible' => true,
            'sensitive' => array_key_exists($key, $sensitive),
        ];
    }
}
