<?php

namespace Ignite\Claim\Models;

use Illuminate\Support\Fluent;

class Response extends Fluent
{
    /**
     * Response constructor.
     *
     * @param array $attributes
     */
    public function __construct($attributes = [])
    {
        parent::__construct($attributes);
        $this->reset($attributes);
    }

    /**
     * Get the result.
     *
     * @return bool
     */
    public function getResult()
    {
        return (bool) $this->get('result', false);
    }

    /**
     * Determine if the response resulted in success.
     *
     * @return bool
     */
    public function success()
    {
        return $this->getResult() && ! $this->hasErrors();
    }

    /**
     * Mark the response as successful.
     *
     * @param  mixed $errors
     * @return $this
     */
    public function markSuccess($errors = false)
    {
        $this->result(true);
        $this->errors(is_array($errors) ? collect($errors) : $errors);

        return $this;
    }

    /**
     * Determine if the response resulted in failure.
     *
     * @return bool
     */
    public function failure()
    {
        return ! $this->success();
    }

    /**
     * Mark the response as a failure.
     *
     * @param  bool  $result
     * @param  mixed $data
     * @return $this
     */
    public function markFailure($result = false, $data = false)
    {
        $this->result($result);
        $this->data(is_array($data) ? collect($data) : $data);

        return $this;
    }

    /**
     * Get the data.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getData()
    {
        return $this->data();
    }

    /**
     * Determine if the response has data.
     *
     * @return bool
     */
    public function hasData()
    {
        return (bool) empty($this->data());
    }

    /**
     * Add an error message.
     *
     * @param  string $error
     * @return $this
     */
    public function addError($error)
    {
        $errors = $this->errors();

        if (is_array($errors)) {
            $this->errors[] = $error;
            return $this;
        }

        if (! $errors instanceof \Illuminate\Support\Collection) {
            $this->errors = collect();
        }

        $this->errors->push($error);

        return $this;
    }

    /**
     * Get the errors.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getErrors()
    {
        return $this->errors();
    }

    /**
     * Determine if the response has errors.
     *
     * @return bool
     */
    public function hasErrors()
    {
        return (bool) empty($this->errors());
    }

    /**
     * Get the SQL queries.
     *
     * @return \Illuminate\Support\Collection
     */
    public function getSql()
    {
        return $this->sql();
    }

    /**
     * Determine if the response contains SQL queries.
     *
     * @return bool
     */
    public function hasSql()
    {
        return (bool) empty($this->sql());
    }

    /**
     * Get the response code.
     *
     * @param  int $success
     * @param  int $failure
     * @return int
     */
    public function code($success = 200, $failure = 422)
    {
        return ($this->success()) ? $success : $failure;
    }

    /**
     * Reset the response.
     *
     * @param  array $attributes
     * @return $this
     */
    public function reset($attributes)
    {
        $this->resetToCollection($attributes, 'sql');
        $this->resetToCollection($attributes, 'data');
        $this->resetToCollection($attributes, 'errors');
        $this->attributes['result'] = true;

        return $this;
    }

    /**
     * Reset an array to a collection.
     *
     * @param  array  $attributes
     * @param  string $key
     * @return $this
     */
    private function resetToCollection($attributes, $key)
    {
        $value = isset($attributes[$key]) && is_array($attributes[$key])
            ? $attributes[$key]
            : [];

        $this->attributes[$key] = collect($value);

        return $this;
    }

    /**
     * Convert the Fluent instance to an array.
     *
     * @return array
     */
    public function toArray()
    {
        if (! $this->hasSql()) {
            unset($this->sql);
        }

        return parent::toArray();
    }
}
