<?php

namespace Ignite\Claim\Models\Grid;

use Ignite\Core\Models\Grid\Formatter;

class ClaimFormatter extends Formatter
{
    /**
     * Custom column for actions.
     *
     * @param  \Ignite\Core\Entities\Participant $model
     * @return \Illuminate\Support\HtmlString
     */
    public function actions($model)
    {
        $actions = [
            [
                'click' => '',
                'class' => '',
                'icon' => 'eye-open',
                'label' => 'View',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.claim.show', [
                    'claimId' => $model->claim_id,
                    'claimParticipantId' => $model->id
                ])
            ],
            [
                'click' => '',
                'class' => '',
                'icon' => 'pencil',
                'label' => 'Edit',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.claim.edit', [
                    'claimId' => $model->claim_id,
                    'userId' => $model->id
                ])],
        ];

        if (in_array($model->status, ['pending', 'cancelled', 'declined'])) {
            $actions[] = [
                'click' => '',
                'icon' => 'trash',
                'label' => 'Delete',
                'click' => 'This action will permanently delete this claim participant record? Are you sure you want to continue?',
                'class' => 'delete-entry',
                'text' => 'text-danger',
                'type' => 'default',
                'link' => route('admin.claim.destroy', [
                    'claimId' => $model->claim_id,
                    'claimParticipantId' => $model->id
                ])
            ];
        }

        return $this->buttons($actions);
    }

    /**
     * Format a claim checkbox.
     *
     * @param $model
     * @return \Illuminate\Support\HtmlString
     */
    public function checkbox($model)
    {
        return $this->formatCheckbox($model->id);
    }

    /**
     * Format the status column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function status($model)
    {
        $label = ucfirst($model->status);
        $colors = [
            'pending' => 'bg-yellow',
            'approved' => 'bg-green',
            'issued' => 'bg-aqua',
            'declined' => 'label-danger',
            'cancelled' => 'bg-red',
        ];

        if (! isset($colors[$model->status])) {
            return $label;
        }

        return $this->html(sprintf(
            '<a href="%s" class="label %s">%s</a>',
            route('admin.claim.list', $model->status),
            $colors[$model->status],
            $label
        ));
    }

    /**
     * Format a participant email with a link to the participant view page.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function participantEmail($model)
    {
        if (empty($model->user_id)) {
            return $model->participant_email;
        }

        return $this->html(sprintf(
            '<a href="%s">%s</a>',
            route('admin.participant.show', $model->user_id),
            $model->participant_email
        ));
    }

    /**
     * Format the date submitted at column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function submittedAt($model)
    {
        return $this->html($this->formatDate($model, 'date_submitted', 'd/m/Y'));
    }

    /**
     * Format the date submitted at column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function activityAt($model)
    {
        return $this->html($this->formatDate($model, 'activity_date', 'd/m/Y', 'Y-m-d'));
    }

    /**
     * Format the revenue column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function revenue($model)
    {
        return $this->formatAmount($model->claim, 'revenue_sold');
    }

    /**
     * Format the value column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function value($model)
    {
        return $this->html($this->formatNumber($model, 'value', 2));
    }

    /**
     * Format the value calculated column.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function valueCalculated($model)
    {
        return $this->html($this->formatNumber($model, 'value_calculated', 2));
    }
}
