<?php

namespace Ignite\Claim\Models\Report;

use Ignite\Core\Entities\User;
use Illuminate\Database\DatabaseManager;
use Ignite\Claim\Models\Grid\AbstractClaimTable;
use Ignite\Claim\Models\Grid\ClaimFormatter;
use Yajra\DataTables\DataTables;

class ClaimReport extends AbstractClaimTable
{
    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [
        'form.claim.partial.claim_fields_external',
        //'form.claim.partial.claim_fields_internal',
    ];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'username'
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'user_id', 'first', 'last', 'lineitems', 'value'
    ];

    /**
     * @var ClaimFormatter
     */
    protected $formatter;

    /**
     * ClaimReport constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param array $params
     * @param ClaimFormatter $formatter
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        array $params = [],
        ClaimFormatter $formatter
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'claim_id' => [
                'title' => 'Claim ID',
                'name' => 'claim_participant.claim_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'user_id' => [
                'title' => 'User ID',
                'name' => 'claim_participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'date_submitted' => [
                'title' => 'Date Entered',
                'name' => 'claim_participant.date_submitted',
                'orderable' => true,
                'exportable' => true,
            ],
            'customer_name' => [
                'title' => 'Customer Name',
                'name' => 'claim.customer_name',
                'orderable' => true,
                'exportable' => true,
            ],
            'deal_registration_date' => [
                'title' => 'Deal Registration',
                'name' => 'claim.deal_registration_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'status' => [
                'title' => 'Status',
                'name' => 'claim_participant.status',
                'orderable' => true,
                'exportable' => true,
            ],
            'lineitems' => [
                'title' => 'Offers',
                'name' => 'lineitems',
                'orderable' => true,
                'exportable' => true,
                'searchable' => false,
            ],
            'value' => [
                'order' => 1000,
                'title' => 'Amount',
                'name' => 'claim_participant.value',
                'searchable' => false,
                'orderable' => true,
                'exportable' => true,
            ]
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = collect($this->getColumns())->keyBy('name')->except(['total', 'lineitems'])->keys()->toArray();
        $query = $this->getConnection()->table('claim_participant')
            ->select($columns)
            ->addSelect($this->getConnection()->raw('GROUP_CONCAT(claim_lineitem.name) as lineitems'))
            ->distinct()
            ->leftJoin('core_user as user', 'user.user_id', '=', 'claim_participant.user_id')
            ->leftJoin('core_participant as participant', 'participant.user_id', '=', 'claim_participant.user_id')
            ->leftJoin('claim', 'claim.id', '=', 'claim_participant.claim_id')
            ->leftJoin('claim_lineitem', 'claim_lineitem.claim_id', '=', 'claim_participant.claim_id')
            ->where('user.internal', User::TYPE_PARTICIPANT)
            ->groupBy($columns);

        return $this->applyScopes($query);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'status' => [$this->formatter, 'status'],
        ];
    }
}
