<?php

namespace Ignite\Claim\Entities;

use Ignite\Claim\Entities\Debug;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Core\Entities\Participant;
use Ignite\Claim\Repositories\OfferRepository;
use Ignite\Claim\Repositories\RuleRepository;

class RuleCalculate
{
    /** @var ClaimRepository */
    private $claimRepository;

    /** @var OfferRepository */
    private $offerRepository;

    /** @var bool|Debug */
    private $debug = false;

    /** @var bool|array */
    private $claim = false;

    /** @var bool */
    private $activeOnly = true;

    /** @var bool */
    private $ruleIds = false;

    /** @var bool */
    private $rules = false;

    /** @var array */
    private $promotionRules = [];

    /** @var bool */
    public $lastParams = false;

    /** @var bool */
    public $response = false;

    /**
     * RuleCalculate constructor.
     *
     * @param ClaimRepository $claimRepository
     * @param OfferRepository $offerRepository
     * @param Debug|null      $debug
     */
    public function __construct(
        ClaimRepository $claimRepository,
        OfferRepository $offerRepository,
        Debug $debug = null
    ) {
        $this->claimRepository = $claimRepository;
        $this->offerRepository = $offerRepository;
        $this->debug = is_null($debug) ? new Debug(0, 10, 10) : $debug;
    }

    /**
     * Initialize the standard response structure.
     *
     * @param  array $params
     * @return void
     */
    protected function initResponse($params)
    {
        $this->lastParams = $params;
        $this->response = new \stdClass();
        $this->response->result = true;
        $this->response->data = new \stdClass;
        $this->response->data->values = 0;
        $this->response->errors = [];
    }

    /**
     * Get the value of a claim.
     *
     *  - Check that the provided parameters pass basic eligibility
     *  - Check that there are lineitems to derive a value
     *  - Calculate the value per lineitem given the current offer and rules
     *  - Calculate the total from the total per lineitem
     *
     * @param  array $params
     * @return bool
     */
    public function value($params)
    {
        $this->initResponse($params);

        // Get the Control Parameters
        $claim = (isset($params['claimId'])) ? $params['claimId'] : false;
        $claim = (isset($params['claim'])) ? $params['claim'] : $claim;

        $this->activeOnly = (isset($params['activeOnly'])) ? $params['activeOnly'] : true;
        $this->debug->debugMode = empty($params['debugMode']) ? $this->debug->debugMode : $params['debugMode'];
        $this->debug->debugLevel = empty($params['debugLevel']) ? $this->debug->debugLevel : $params['debugLevel'];
        $this->debug->detailLevel = empty($params['detailLevel']) ? $this->debug->detailLevel : $params['detailLevel'];

        if (! empty($params['ruleIds'])) {
            if (is_array($params['ruleIds'])) {
                $this->ruleIds = $params['ruleIds'];
            } elseif (is_string($params['ruleIds'])) {
                $this->ruleIds = preg_split('/\s*[,;\n]\s*/', trim($params['ruleIds']));
            } else {
                $this->ruleIds = false;
            }
        }

        // Get the Claim we need to calculate for
        // @todo Consider passing the claim eloquent model as a dependency
        // @codeCoverageIgnoreStart
        if (is_int($claim) || is_string($claim)) {
            $claimId = intval($claim);
            $response = $this->claimRepository->get([
                'class' => 'claim',
                'action' => 'find',
                'mode' => 'array',
                'with' => ['lineitems', 'participant'],
                'id' => $claimId
            ]);

            if (! $response->result) {
                $this->response->result = false;
                $this->response->errors[] = sprintf("Can't find Claim for Id = '%s'.", $claimId);
                $this->response->errors = array_merge($this->response->errors, $response->errors);
                return $this->response;
            }

            $claim = $response->data;
        } elseif (is_object($claim)) {
            $claim = (array) $claim;
        }
        // @codeCoverageIgnoreEnd

        if (! is_array($claim)) {
            $this->response->result = false;
            $this->response->errors[] = sprintf("Invalid or missing Claim Id and/or Claim Data.");
            return $this->response;
        }

        if (! isset($claim['offer_promotion_id']) || empty($claim['offer_promotion_id'])) {
            throw new \DomainException(
                'No offer promotion id was provided. ' .
                'Rule calculation is dependant upon knowing which promotion offer to check against.'
            );
        }

        $this->claim = $claim;
        $promotionId = $this->claim['offer_promotion_id'];

        // Get Rules to use in Calculating the Value
        if (! $this->getRules($this->ruleIds, $promotionId)) {
            return $this->response;
        }

        // Determine Participant Type we are Calculating the Value for
        if (! empty($params['participantType'])) {
            // Check if Passed in
            $participantType = $params['participantType'];
        } elseif (! empty($this->claim['participant']['participant_type'])) {
            // Check if From Claim Participant Record
            $participantType = $this->claim['participant']['participant_type'];
        } elseif (! empty($this->claim['participant']['type'])) {
            // Check if From Participant Record, ie: the Submitter
            $participantType = $this->claim['participant']['type'];
        } else {
            $this->response->result = false;
            $this->response->errors[] = sprintf("Unable to determine Participant Type.");
            return $this->response;
        }

        // Do the Calculation Process
        $lineItemValues = $this->processClaim($participantType);

        // Assemble Response Data
        $this->response->result = true;
        $this->response->data = new \stdClass;
        $this->response->data->values = $lineItemValues;
        $this->response->data->logs = $this->debug->getDebugLog();

        return $this->response;
    }

    /**
     * Normalize the input for comparison based on its type.
     *
     * @param  string $value
     * @param  mixed $type
     *
     * @return mixed
     */
    private function normalize($value, $type)
    {
        $normalValue = $value;

        switch ($type) {
            case 'date':
            case 'datetime':
                $normalValue = strtotime($value);
                break;
        }

        return $normalValue;
    }

    /**
     * Compare the provided data with the rule conditions
     *
     * @param  string $directive
     * @param  mixed  $dbValue
     * @param  mixed  $data1
     * @param  mixed  $data2
     * @return bool
     */
    private function compare($directive, $dbValue, $data1, $data2)
    {
        $result = false;

        switch ($directive) {
            case '=':
                if ($dbValue == $data1) {
                    $result = true;
                }
                break;

            case '!=':
                if ($dbValue != $data1) {
                    $result = true;
                }
                break;

            case '>':
                if ($dbValue > $data1) {
                    $result = true;
                }
                break;

            case '>=':
                if ($dbValue >= $data1) {
                    $result = true;
                }
                break;

            case '<':
                if ($dbValue < $data1) {
                    $result = true;
                }
                break;

            case '<=':
                if ($dbValue <= $data1) {
                    $result = true;
                }
                break;

            case 'BETWEEN':
                if ($dbValue >= $data1 && $dbValue <= $data2) {
                    $result = true;
                }
                break;

            case 'IN':
                $list = preg_split('~(\r\n|[|;\r\n])~', $data1);
                if (in_array($dbValue, $list)) {
                    $result = true;
                }
                break;

            case 'NOT IN':
                $list = preg_split('~(\r\n|[|;\r\n])~', $data1);
                if (! in_array($dbValue, $list)) {
                    $result = true;
                }
                break;

            case 'LIKE':
                $pattern = '#' . $data1 . '#';
                if (preg_match($pattern, $dbValue)) {
                    $result = true;
                }
                break;

            default:
                $result = false;
                break;
        }

        return $result;
    }

    /**
     * Check if the condition is eligible to be applied to the claim participant.
     *
     * @param  array $rule
     * @param  array $condition
     * @return bool
     */
    private function checkCondition($rule, $condition)
    {
        list($table, $column) = explode('|', $condition['db_column']);

        $type = null;
        $table = trim($table);
        $column = trim($column);
        $dbValue = '';

        switch ($table) {
            case 'claim':
                $field = Claim::getFields($column);
                $type = $field['type'];
                if (isset($this->claim[$column])) {
                    $dbValue = $this->normalize($this->claim[$column], $type);
                }
                break;

            case 'participant':
                $field = ClaimParticipant::getFields($column);
                $type = $field['type'];
                if (isset($this->claim['participant'][$column])) {
                    $dbValue = $this->normalize($this->claim['participant'][$column], $type);
                }
                break;

            default:
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Condition Check Failed, unknown Table '%s'",
                    $rule['id'],
                    $rule['name'],
                    $table
                ));

                return false;
                break;
        }

        $data1 = $this->normalize($condition['data_1'], $type);
        $data2 = $this->normalize($condition['data_2'], $type);

        return $this->compare($condition['directive'], $dbValue, $data1, $data2);
    }

    /**
     * Perform the basic minimum checks to determine if the rule should be applied.
     *
     * @param array $rule
     * @return bool
     */
    private function checkRuleBasics($rule)
    {
        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Basic Checks:",
            $rule['id'],
            $rule['name']
        ));

        if (! empty($this->ruleIds)) {
            $ruleList = implode(',', $this->ruleIds);
            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Basic Checks: Rule List Provided: '%s' .",
                $rule['id'],
                $rule['name'],
                $ruleList
            ));
            // @todo Consider removing the following block, since any rules not in ruleIds are filtered out in getRules()
            // @codeCoverageIgnoreStart
            if (! in_array($rule['id'], $this->ruleIds)) {
                $this->debug->logDebug(3, sprintf(
                    "Rule Id %s: '%s', Basic Checks: Rule Id not in List '%s', Failed.",
                    $rule['id'],
                    $rule['name'],
                    $ruleList
                ));

                return false;
            }
            // @codeCoverageIgnoreEnd
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Basic Checks: Rule Id matched List '%s'.",
                $rule['id'],
                $rule['name'],
                $ruleList
            ));
        }

        if ($this->activeOnly && $rule['status'] != 1) {
            $this->debug->logDebug(3, sprintf(
                "Rule Id %s: '%s', Basic Checks: Status Active Only Failed, Status: '%s'.",
                $rule['id'],
                $rule['name'],
                (($rule['status']) ? 'ACTIVE' : 'INACTIVE')
            ));

            return false;
        }

        $this->debug->logDebug(5, sprintf(
            "Rule Id %s: '%s', Basic Checks: Status %s Passed: '%s'.",
            $rule['id'],
            $rule['name'],
            (($this->activeOnly) ? 'Active Only' : 'ALL'),
            (($rule['status']) ? 'ACTIVE' : 'INACTIVE')
        ));

        $createdDateSecs = strtotime($this->claim['created_at']);
        $createdDate     = date('m/d/Y', $createdDateSecs);

        if (! empty($rule['start_date'])) {
            $ruleStartDateSecs = strtotime($rule['start_date']);
            $ruleStartDate     = date('m/d/Y', $ruleStartDateSecs);
            if ($createdDateSecs < $ruleStartDateSecs) {
                $this->debug->logDebug(3, sprintf(
                    "Rule Id %s: '%s', Basic Checks: Start Date '%s' >= '%s', Failed.",
                    $rule['id'],
                    $rule['name'],
                    $createdDate,
                    $ruleStartDate
                ));

                return false;
            }
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Basic Checks: Start Date Passed: '%s' >= '%s'.",
                $rule['id'],
                $rule['name'],
                $createdDate,
                $ruleStartDate
            ));
        }

        if (! empty($rule['end_date'])) {
            $ruleEndDateSecs = strtotime($rule['end_date']);
            $ruleEndDate     = date('m/d/Y', $ruleEndDateSecs);
            if ($createdDateSecs >= $ruleEndDateSecs) {
                $this->debug->logDebug(3, sprintf(
                    "Rule Id %s: '%s', Basic Checks: End Date '%s' < '%s', Failed.",
                    $rule['id'],
                    $rule['name'],
                    $createdDate,
                    $ruleEndDate
                 ));

                return false;
            }
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Basic Checks: End Date Passed: '%s' < '%s'.",
                $rule['id'],
                $rule['name'],
                $createdDate,
                $ruleEndDate
            ));
        }

        return true;
    }

    /**
     * Check if the rule should apply only to specific participants.
     *
     * @param  array  $rule
     * @param  string $participantType
     * @return bool
     */
    private function checkParticipants($rule, $participantType)
    {
        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Participants Check:",
            $rule['id'],
            $rule['name']
        ));

        // -----------------------------
        // Check if ALL Participants allowed
        // -----------------------------
        // $participantType = $this->claim['participant']['type'];

        if ($rule['participants_mode_id'] == 1) {
            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Participants Check: Mode ALL, matched '%s'.",
                $rule['id'],
                $rule['name'],
                $participantType
            ));

            return true;
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Participants Check: Mode By Participant.",
            $rule['id'],
            $rule['name']
        ));

        // -----------------------------
        // Check if Specific Participant allowed
        // -----------------------------
        foreach ($rule['participants'] as $participant) {
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Participants Check: '%s' == '%s'",
                $rule['id'],
                $rule['name'],
                $participant['participant_type'],
                $participantType
            ));
            if ($participant['participant_type'] == $participantType) {
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Participants Check: matched '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $participant['participant_type']
                ));

                return true;
            }
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Participants Check: Type '%s' Not in List.",
            $rule['id'],
            $rule['name'],
            $participantType
        ));

        return false;
    }

    /**
     * Check if the rule should apply only to specific offers.
     *
     * @param $rule
     * @param $lineitem
     * @param $participantType
     *
     * @return bool
     */
    private function checkOffers($rule, $lineitem, $participantType)
    {
        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Offers Check:",
            $rule['id'],
            $rule['name']
        ));

        // -----------------------------
        // Check if ALL Offers allowed
        // -----------------------------
        if ($rule['offers_mode_id'] == 1) {
            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Offers Check: Mode ALL, matched Type '%s'.",
                $rule['id'],
                $rule['name'],
                $participantType
            ));

            return true;
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Offers Check: Mode By Participant.",
            $rule['id'],
            $rule['name']
        ));

        // -----------------------------
        // Check if Specific Offer allowed
        // -----------------------------
        foreach ($rule['offers'] as $offer) {
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Offers Check: '%s' == '%s'",
                $rule['id'],
                $rule['name'],
                $offer['offer_id'],
                $lineitem['offer_id']
            ));
            if ($offer['offer_id'] == $lineitem['offer_id']) {
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Offers Check: matched Id '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $lineitem['offer_id']
                ));

                return true;
            }
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Offers Check: Offer Id '%s' Not in List.",
            $rule['id'],
            $rule['name'],
            $lineitem['offer_id']
        ));

        return false;
    }

    /**
     * Check if the rule should apply only if the given conditions match in the form of boolean AND.
     *
     * @param  array $rule
     * @return bool
     */
    private function checkConditions($rule)
    {
        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Conditions Checks:",
            $rule['id'],
            $rule['name']
        ));

        // -----------------------------
        // Check if NO Conditions
        // -----------------------------
        if (count($rule['conditions']) == 0) {
            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Conditions Check: No Conditions, matched.",
                $rule['id'],
                $rule['name']
            ));

            return true;
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Conditions Check: has %s conditions:",
            $rule['id'],
            $rule['name'],
            count($rule['conditions'])
        ));

        // -----------------------------
        // Check Matches All Conditions
        // -----------------------------
        foreach ($rule['conditions'] as $condition) {
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Condition Check: testing '%s' '%s' '%s' '%s'.",
                $rule['id'],
                $rule['name'],
                $condition['db_column'],
                $condition['directive'],
                $condition['data_1'],
                $condition['data_2']
            ));
            if (! $this->checkCondition($rule, $condition)) {
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Condition Check: '%s' '%s' '%s' '%s' FALSE.",
                    $rule['id'],
                    $rule['name'],
                    $condition['db_column'],
                    $condition['directive'],
                    $condition['data_1'],
                    $condition['data_2']
                ));

                return false;
            }
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Condition Check: testing '%s' '%s' '%s' '%s' TRUE.",
                $rule['id'],
                $rule['name'],
                $condition['db_column'],
                $condition['directive'],
                $condition['data_1'],
                $condition['data_2']
            ));
        }

        return true;
    }

    /**
     * Determine if the rule is eligible to be applied.
     *
     * @param  array  $rule
     * @param  array  $lineitem
     * @param  string $participantType
     * @return bool
     */
    private function checkRuleEligibility($rule, $lineitem, $participantType)
    {
        if (! $this->checkParticipants($rule, $participantType)) {
            $this->debug->logDebug(3, sprintf(
                "Rule Id %s: '%s', Participant Checks Failed.",
                $rule['id'],
                $rule['name']
            ));

            return false;
        }

        if (! $this->checkOffers($rule, $lineitem, $participantType)) {
            $this->debug->logDebug(3, sprintf(
                "Rule Id %s: '%s', Offers Checks Failed.",
                $rule['id'],
                $rule['name']
            ));

            return false;
        }

        if (! $this->checkConditions($rule)) {
            $this->debug->logDebug(3, sprintf(
                "Rule Id %s: '%s', Conditions Checks Failed.",
                $rule['id'],
                $rule['name']
            ));

            return false;
        }

        return true;
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param $rule
     * @param $participantType
     *
     * @return bool
     */
    private function getRuleParticipantValue($rule, $participantType)
    {
        // If the Rule is a Single Value mode, we need to Dummy up a Participant Value record to return
        // @todo Remove the following code since it is never called with the context it checks for.
        // @codeCoverageIgnoreStart
        if ($rule['value_mode_id'] == 2 || $rule['value_mode_id'] == 4) {
            $participantValue = [];
            $participantValue['offer_id']         = 0;
            $participantValue['participant_type'] = 'ALL';
            $participantValue['value_type_id']    = $rule['value_type_id'];
            $participantValue['value']            = $rule['value'];
            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Rule Value Mode ALL, matched for Participant Type '%s'.",
                $rule['id'],
                $rule['name'],
                $participantType
            ));

            return $participantValue;
        }
        // @codeCoverageIgnoreEnd

        $this->debug->logDebug(4, sprintf(
             "Rule Id %s: '%s', Value Mode NOT ALL, checking Rule Values for Type '%s':",
             $rule['id'],
             $rule['name'],
             $participantType
         ));

        // Try to find a matching Rule Participant Value record
        foreach ($rule['values'] as $participantValue) {
            $this->debug->logDebug(5, sprintf(
                "Rule Id %s: '%s', Checking Rule Value Id %s: '%s', for Participant Type '%s'.",
                $rule['id'],
                $rule['name'],
                $participantValue['id'],
                $participantValue['participant_type'],
                $participantType
            ));
            if ($participantValue['participant_type'] == $participantType) {
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Rule Value Id %s: '%s' matched for Participant Type '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $participantValue['id'],
                    $participantValue['participant_type'],
                    $participantType
                 ));

                return $participantValue;
            }
        }

        // @todo Remove the following code since it is never called.
        // @codeCoverageIgnoreStart
        $this->debug->logDebug(3, sprintf(
             "Rule Id %s: '%s', no Rule value for Participant Type '%s'.",
             $rule['id'],
             $rule['name'],
             $participantType
         ));

        return false;
        // @codeCoverageIgnoreEnd
    }

    /**
     * @todo: Explain the functionality.
     *
     * @param $rule
     * @param $lineitem
     * @param $participantType
     *
     * @return bool
     */
    private function getOfferValue($rule, $lineitem, $participantType)
    {
        foreach ($rule['offers'] as $offer) {
            // ------------------------------------------
            // Skip Offers Mode if we don't have a matching Rule Offer Id
            // ------------------------------------------
            if ($offer['offer_id'] != $lineitem['offer_id']) {
                $this->debug->logDebug(5, sprintf(
                    "Rule Id %s: '%s', Offer Id '%s' no Offer Id match, skipped...",
                    $rule['id'],
                    $rule['name'],
                    $offer['offer_id']
                 ));
                continue;
            }

            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Offer Id '%s' matched, checking for Participant Type '%s'.",
                $rule['id'],
                $rule['name'],
                $offer['offer_id'],
                $participantType
            ));

            // ------------------------------------------
            // If the Offer is a Single Value mode, we need to Dummy up a Participant Value record to return
            // ------------------------------------------
            if ($offer['offer']['value_mode_id'] == 1 || empty($offer['offer']['value_mode_id'])) {
                $participantValue                     = [];
                $participantValue['offer_id']         = $offer['id'];
                $participantValue['participant_type'] = 'ALL';
                $participantValue['value_type_id']    = $offer['offer']['value_type_id'];
                $participantValue['value']            = $offer['offer']['value'];
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Offer '%s' Value Mode ALL, matched for Participant Type '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $offer['offer_id'],
                    $participantType
                ));

                return $participantValue;
            }

            $this->debug->logDebug(4, sprintf(
                "Rule Id %s: '%s', Offer '%s' Value Mode NOT ALL, checking Offer Values for Type '%s':",
                $rule['id'],
                $rule['name'],
                $offer['offer_id'],
                $participantType
            ));

            // ------------------------------------------
            // Try to find a matching Offer Participant Value record
            // ------------------------------------------
            if ($offer['offer']['value_mode_id'] == 2) {
                foreach ($offer['values'] as $participantValue) {
                    $this->debug->logDebug(5, sprintf(
                        "Rule Id %s: '%s', Checking Offer Id '%s' Value Id %s: '%s', for Participant Type '%s'.",
                        $rule['id'],
                        $rule['name'],
                        $offer['offer_id'],
                        $participantValue['id'],
                        $participantValue['participant_type'],
                        $participantType
                    ));
                    if ($participantValue['participant_type'] == $participantType) {
                        $this->debug->logDebug(4, sprintf(
                            "Rule Id %s: '%s', Offer Id '%s' Value Id %s: '%s', matched for Participant Type '%s'.",
                            $rule['id'],
                            $rule['name'],
                            $offer['offer_id'],
                            $participantValue['id'],
                            $participantValue['participant_type'],
                            $participantType
                        ));

                        return $participantValue;
                    }
                }
                $this->debug->logDebug(4, sprintf(
                    "Rule Id %s: '%s', Offer Id '%s' no Value for Participant Type '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $offer['offer_id'],
                    $participantType
                ));
            }
        }

        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', no Offer Value for Participant Type '%s'.",
            $rule['id'],
            $rule['name'],
            $participantType
        ));

        return false;
    }

    /**
     * Get the value, type and basis for calculation per rule per line item per participant type
     *
     * @param $rule
     * @param $lineitem
     * @param $participantType
     *
     * @return array
     */
    private function getValueAndRules($rule, $lineitem, $participantType)
    {
        $valueAndRules = [];
        $valueSourceId = $rule['value_mode_id'];
        $valueSourceText = (! empty(Rule::$valueSourceOptions[$valueSourceId]))
            ? Rule::$valueSourceOptions[$valueSourceId]
            : 'Unknown Source Type';

        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Value Source is '%s':",
            $rule['id'],
            $rule['name'],
            $valueSourceText
        ));

        switch ($valueSourceId) {
            // From Offer
            case 1:
                $offerValue = $this->getOfferValue($rule, $lineitem, $participantType);
                $valueAndRules['value'] = $offerValue['value'];
                $valueAndRules['type'] = $offerValue['value_type_id'];
                $valueAndRules['basis'] = 'new';
                break;

            // Rule Single Value
            case 2:
                $valueAndRules['value'] = $rule['value'];
                $valueAndRules['type'] = $rule['value_type_id'];
                $valueAndRules['basis'] = 'new';
                break;

            // Rule By Participant
            case 3:
                $participantValue = $this->getRuleParticipantValue($rule, $participantType);
                $valueAndRules['value'] = $participantValue['value'];
                $valueAndRules['type'] = $participantValue['value_type_id'];
                $valueAndRules['basis'] = 'new';
                break;

            // Running Balance Rule Single Value
            case 4:
                $valueAndRules['value'] = $rule['value'];
                $valueAndRules['type'] = $rule['value_type_id'];
                $valueAndRules['basis'] = 'balance';
                break;

            // Running Balance By Participant
            case 5:
                $participantValue = $this->getRuleParticipantValue($rule, $participantType);
                $valueAndRules['value'] = $participantValue['value'];
                $valueAndRules['type'] = $participantValue['value_type_id'];
                $valueAndRules['basis'] = 'balance';
                break;

            default:
                $this->debug->logDebug(0, sprintf(
                    "ERROR: Rule Id %s: '%s', Source '%s'.",
                    $rule['id'],
                    $rule['name'],
                    $valueSourceText
                ));
                $valueAndRules['value'] = 0;
                $valueAndRules['type']  = 'amount';
                $valueAndRules['basis'] = 'new';
                break;
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Source '%s' Value '%s', Type '%s', Basis '%s'.",
            $rule['id'],
            $rule['name'],
            $valueSourceText,
            $valueAndRules['value'],
            $valueAndRules['type'],
            $valueAndRules['basis']
        ));

        return $valueAndRules;
    }

    /**
     * Calculate the value for the rule given a lineitems
     *
     * @param array  $rule
     * @param array  $lineitem         Required in order to get the lineitem quantity
     * @param array  $valueAndRules
     * @param string $basis
     * @return float|int
     */
    private function getValue($rule, $lineitem, $valueAndRules, $basis)
    {
        $type = ucfirst($valueAndRules['type']);
        $typeText = (! empty(Rule::$valueTypeOptions[$type]))
            ? Rule::$valueTypeOptions[$type]
            : sprintf("Unknown Value Type `%s`", $type);

        if ($valueAndRules['type'] == 2) {
            // Check if Value Type is a Percent
            $value = ($basis * $valueAndRules['value']) / 100;
        } else {
            // Default Value type to Amount
            $value = $valueAndRules['value'];
            // Multiply the Value by the Quantity in the Line Item
            $value *= $lineitem['qty'];
        }

        $this->debug->logDebug(4, sprintf(
            "Rule Id %s: '%s', Source '%s', Basis '%s', Type '%s', Amount '%s', Qty '%s', Value '%s'.",
            $rule['id'],
            $rule['name'],
            $valueAndRules['basis'],
            $basis,
            $typeText,
            $valueAndRules['value'],
            $lineitem['qty'],
            $value
         ));

        return $value;
    }

    /**
     * @todo: Wrap this with a test to cover the different paths through the code.
     *
     * @param $rule
     * @param $lineitem
     * @param $participantType
     * @param $balance
     *
     * @return bool|float|int
     */
    private function calculateRuleValue($rule, $lineitem, $participantType, $balance)
    {
        $valueAndRules = $this->getValueAndRules($rule, $lineitem, $participantType);

        // @todo Consider removing the following block because it will never be called given the structure of the class
        // @codeCoverageIgnoreStart
        if (empty($valueAndRules)) {
            $this->debug->logDebug(3, sprintf(
                "Rule Id %s: '%s', Line Item %s: '%s', unable to get Value.",
                $rule['id'],
                $rule['name'],
                $lineitem['id'],
                $lineitem['name']
            ));

            return false;
        }
        // @codeCoverageIgnoreEnd

        // ---------------------------------------
        // Determine Basis Source for Percent Calculations
        // ---------------------------------------
        if ($valueAndRules['basis'] == 'balance') {
            $basis     = $balance;
            $basisText = 'balance';
        } else {
            // Note: This could be an issue, as the Sale Value is not always on the Claim Entry Form. Therefore, it
            // would not exist here on an Edit => Save. This would technically be a User Configuration error, but we
            // don't want the code to throw an exception because of it.
            $basis     = (isset($this->claim['sale_value'])) ? $this->claim['sale_value'] : 0;
            $basisText = 'claim.sale_value';
        }

        // Check for Zero Basis when Value Type is a Percentage
        if ($valueAndRules['type'] == 2 && $basis == 0) {
            $this->debug->logDebug(1, sprintf(
                "WARNING: Rule Id %s: '%s', Line Item %s: '%s', %s basis is Zero.",
                $rule['id'],
                $rule['name'],
                $lineitem['id'],
                $lineitem['name'],
                $basisText
            ));
        }

        $value = $this->getValue($rule, $lineitem, $valueAndRules, $basis);

        $this->debug->logDebug(3, sprintf(
            "Rule Id %s: '%s', Line Item %s: '%s', Value is '%s':",
            $rule['id'],
            $rule['name'],
            $lineitem['id'],
            $lineitem['name'],
            $value
        ));

        return $value;
    }

    /**
     * Process all available rules for the claim participant on a lineitem by lineitem basis.
     *
     * @param  array $lineitem
     * @param  string $participantType
     * @return array
     */
    private function processRules($lineitem, $participantType)
    {
        $balance    = 0;
        $ruleValues = [];

        // ------------------------------------------
        // Process each Rule for each Line Item
        // ------------------------------------------
        foreach ($this->rules as $index => $rule) {
            if ($rule['disabled']) {
                $this->debug->logDebug(6, sprintf(
                    "Rule Id %s: '%s', Disabled in Basic Checks, skipped...",
                    $rule['id'],
                    $rule['name']
                ));
                continue;
            }

            $this->debug->logDebug(3, sprintf(
                "Checking Rule Id %s: '%s'.",
                $rule['id'],
                $rule['name']
            ));

            // --------------------------------
            // Check if we have a limit on the number of times this Rule can be applied to a Claim
            // --------------------------------
            if (
                $this->rules[$index]['max_times'] > 0 &&
                $this->rules[$index]['appliedCount'] >= $this->rules[$index]['max_times']
            ) {
                $this->debug->logDebug(2, sprintf(
                    "Rule Id %s: '%s', Max Limit of %s reached, skipped...",
                    $rule['id'],
                    $rule['name'],
                    $this->rules[$index]['max_times']
                ));
                continue;
            }

            // --------------------------------
            // Check Rule eligibility for Line Item
            // --------------------------------
            if (! $this->checkRuleEligibility($rule, $lineitem, $participantType)) {
                $this->debug->logDebug(2, sprintf(
                    "Rule Id %s: '%s', Rule Checks Failed, skipped...",
                    $rule['id'],
                    $rule['name']
                ));
                continue;
            }

            // --------------------------------
            // Calculate Rule Value for Line Item
            // --------------------------------
            $this->debug->logDebug(3, sprintf(
                "Checking Rule Id %s: '%s', Rule Checks Passed, calculating value:",
                $rule['id'],
                $rule['name']
            ));

            $value = $this->calculateRuleValue($rule, $lineitem, $participantType, $balance);

            if ($value !== false) {
                $balance += $value;
                $ruleKey = 'rule:' . $rule['id'] . ' - ' . $rule['name'];
                $ruleValues[$ruleKey] = $value;
                ++$this->rules[$index]['appliedCount'];
            }
        }

        $ruleValues['total'] = $balance;

        return $ruleValues;
    }

    /**
     * Determine the claim value for the provided participant type.
     *
     * @param  string $participantType
     * @return array
     */
    private function processClaim($participantType)
    {
        $claimValue = 0;
        $lineItemValues = [];
        $lineItemValues['claim_total'] = 0;

        if (! empty($this->claim['participant']['first']) && ! empty($this->claim['created_at'])) {
            $this->debug->logDebug(1, sprintf(
                "Processing Claim Id %s, Date %-10.10s, User %s: %s.",
                $this->claim['id'],
                $this->claim['created_at'],
                $this->claim['participant']['user_id'],
                $this->claim['participant']['first'] . ' ' . $this->claim['participant']['last']
            ));
        } else {
            $this->debug->logDebug(1, sprintf("Processing Claim, Date %s", date('Y-m-d')));
        }

        // Check for No Line Items on the Claim
        if (empty($this->claim['lineitems'])) {
            $this->response->result = false;
            $this->debug->logDebug(1, sprintf("WARNING: No Line Items for Claim, nothing to do...."));
            return $lineItemValues;
        }

        // Pre Check All Rules for Basic eligibility
        // Disable for all future checks if they fail here.
        $countDisabled = 0;
        foreach ($this->rules as $index => $rule) {
            $this->debug->logDebug(2, sprintf(
                "Checking Rule Id %s: '%s'.",
                $rule['id'],
                $rule['name']
            ));
            $this->rules[$index]['disabled']     = false;
            $this->rules[$index]['appliedCount'] = 0;
            if (! $this->checkRuleBasics($rule)) {
                $this->debug->logDebug(2, sprintf(
                    "Rule Id %s: '%s', Basic Checks Failed, Disabled...",
                    $rule['id'],
                    $rule['name']
                ));
                $this->rules[$index]['disabled'] = true;
                ++$countDisabled;
            }
        }

        if ($countDisabled == count($this->rules)) {
            $this->response->result = false;
            $this->debug->logDebug(1, "WARNING: All Rules Disabled, nothing to do...");
            return $lineItemValues;
        }

        // Process each Line Item separately
        $lineCount = 1;
        foreach ($this->claim['lineitems'] as $lineitem) {
            $this->debug->logDebug(2, sprintf(
                "Processing Line %s Item %s, Offer Id %s, Offer Name '%s', Qty '%s'.",
                $lineCount,
                $lineitem['id'],
                $lineitem['offer_id'],
                $lineitem['name'],
                $lineitem['qty']
            ));
            $ruleValues = $this->processRules($lineitem, $participantType);

            // Organize the Values by Line Item, by Rule
            $lineItemKey = 'lineitem:' . $lineitem['id'] . ' - ' . $lineitem['name'];
            foreach ($ruleValues as $ruleKey => $value) {
                $lineItemValues[$lineItemKey][$ruleKey] = $value;
            }
            $claimValue += $ruleValues['total'];
        }

        $lineItemValues['claim_total'] = $claimValue;

        return $lineItemValues;
    }

    /**
     * Get all the promotion offers that are related to this rule.
     *
     * @param  array $rule
     * @param  int   $promotionId
     * @return bool|\stdClass
     */
    private function getAllPromotionOffers($rule, $promotionId)
    {
        // Get All Offers for the Promotion Id on this Rule
        $response = $this->offerRepository->get([
            'class' => 'claim_offer',
            'action' => 'getList',
            'mode' => 'array',
            'with' => ['values'],
            'promotion_id' => $promotionId
        ]);

        if (! $response->result) {
            return $response;
        }

        $offers = $response->data;
        // Build Individual Rule Offers for each Promotion Offer
        $response         = new \stdClass;
        $response->result = true;
        $response->data   = false;
        $response->errors = [];

        $allRuleOffers = [];
        foreach ($offers as $offer) {
            $ruleOffer             = [];
            $ruleOffer['id']       = 'ALL';
            $ruleOffer['rule_id']  = $rule['id'];
            $ruleOffer['offer_id'] = $offer['id'];
            $ruleOffer['values']   = $offer['values'];

            unset($offer['values']);
            $ruleOffer['offer'] = $offer;
            $allRuleOffers[] = $ruleOffer;
        }

        $response->data = $allRuleOffers;

        return $response;
    }

    /**
     * Find the list of rules for the current promotion.
     *
     * @param  array $ruleIds
     * @param  int   $promotionId
     * @return bool
     */
    private function getRules($ruleIds, $promotionId)
    {
        if (! isset($this->promotionRules[$promotionId])) {
            $ruleSvc = app(RuleRepository::class);
            $params = [
                'class' => 'claim_rule',
                'action' => 'getList',
                'mode' => 'array',
                'with' => ['participants', 'offers.offer', 'offers.values', 'values', 'conditions'],
                'offer_promotion_id' => $promotionId,
                'orderBy' => ['order']
            ];
            $response = $ruleSvc->get($params);

            // Unable to cover this code because we can't simulate a realistic database error.
            // @codeCoverageIgnoreStart
            if (! $response->result) {
                $this->response->result = false;
                $this->response->errors[] = sprintf("Unable to get Rules for Promotion Id = '%s'.", $promotionId);
                $this->response->errors = array_merge($this->response->errors, $response->errors);
                return false;
            }
            // @codeCoverageIgnoreEnd

            $this->promotionRules[$promotionId] = $response->data;
        }

        $rules = $this->promotionRules[$promotionId];

        // Check if we need to limit the Rules to a list provided
        // Also, put them in the order of the list provided
        if (! empty($ruleIds)) {
            $rules = [];
            foreach ($ruleIds as $id) {
                foreach ($response->data as $rule) {
                    if ($rule['id'] == $id) {
                        $rules[] = $rule;
                    }
                }
            }
        }

        // Check if ALL Offers is set for the Rule Offers mode
        foreach ($rules as $index => $tmpRule) {
            $rule = &$rules[$index];

            if ($rule['offers_mode_id'] == 1) {
                $response = $this->getAllPromotionOffers($rule, $promotionId);

                // Unable to cover this code because we can't simulate a realistic database error.
                // @codeCoverageIgnoreStart
                if (! $response->result) {
                    $this->response->result = false;
                    $this->response->errors[] = sprintf(
                        "Unable to get ALL Offers for Promotion Id = '%s'.",
                        $promotionId
                    );
                    $this->response->errors = array_merge($this->response->errors, $response->errors);

                    return false;
                }
                // @codeCoverageIgnoreEnd
                $rule['offers'] = $response->data;
            }
        }

        $this->rules = $rules;

        if (empty($this->rules)) {
            $this->response->result = false;
            if (! empty($ruleIds)) {
                $this->response->errors[] = sprintf(
                    "No matching Rules in list (%s), for Promotion Id = '%s'.",
                    implode(',', $ruleIds),
                    $promotionId
                );
            } else {
                $this->response->errors[] = sprintf("No Rules for Promotion Id = '%s'.", $promotionId);
            }
            $this->response->errors = array_merge($this->response->errors, $response->errors);

            return false;
        }

        return true;
    }
}
