<?php

namespace Ignite\Claim\Traits;

use Ignite\Claim\Models\Legacy\TableConfigService;

trait ServiceProviderCommon
{
    /**
     * Check if Params or Data is an array of Params or Data
     *
     * @param  array $params
     * @return bool
     */
    public static function isArray($params)
    {
        $first = array_first($params);

        return is_array($first);
    }

    /**
     * Return a Single Params array from an Array of Params or Single Params
     *
     * @param  array $params
     * @return array
     */
    public static function getSingleParams($params)
    {
        return self::isArray($params) ? $params[0] : $params;
    }

    /**
     * Return a value from a Params, or the first of an Array of Params
     *
     * @param  string $name
     * @param  array  $params
     * @return mixed
     */
    public static function getValueFromParams($name, $params)
    {
        $checkParams = self::getSingleParams($params);

        return (! empty($checkParams[$name])) ? $checkParams[$name] : false;
    }

    /**
     * Return the action from a Params, or the first of an Array of Params
     *
     * @param  array $params
     * @return string
     */
    public static function getActionFromParams($params)
    {
        return self::getValueFromParams('action', $params);
    }

    /**
     * Return the class from a Params, or the first of an Array of Params
     *
     * @param  array $params
     * @return string
     */
    public static function getClassFromParams($params)
    {
        return self::getValueFromParams('class', $params);
    }

    /**
     * Return the !<>= from the beginning of a Filter field. ie: '>=start_date'
     *
     * @param  mixed $fieldValue
     * @param  array $filterParams
     * @return array
     */
    public static function getOperatorValue($fieldValue, $filterParams)
    {
        $operator = (! empty($filterParams['operator'])) ? $filterParams['operator'] : '=';
        $rc = preg_match(
            '~([\!\=\<\>]+|^like:|^null:|^isnull:|^notnull:|^between:|^in:|^isin:|^notin:)~i',
            $fieldValue,
            $match
        );

        if ($rc) {
            $operator = $match[0];
            $value = preg_replace("~^$operator~", '', $fieldValue);
            $operator = strtolower(rtrim($operator, ':'));
        } else {
            $value = $fieldValue;
        }

        return [$operator, $value];
    }

    /**
     * Format Value into DB acceptable format
     *
     * @param  mixed  $value
     * @param  string $type
     * @return mixed
     */
    public static function dbFormat($value, $type)
    {
        $formatted = $value;

        switch ($type) {
            case 'date':
            case 'datetime':
                if (trim($value) == '') {
                    $formatted = null;
                } else {
                    $format = ($type == 'datetime') ? 'Y-m-d H:i:s' : 'Y-m-d';
                    $formatted = date($format, strtotime($value));
                }
                break;
        }

        return $formatted;
    }

    /**
     * Initialize the Standard Response Structure
     *
     * @param  array $params
     * @return void
     */
    protected function initResponse($params)
    {
        $this->lastParams = $params;
        $this->processed = false;
        $this->response = new \stdClass();
        $this->response->result = true;
        $this->response->data = false;
        $this->response->errors = [];

        if (! empty($params['withSql'])) {
            $this->response->sql = [];
        }
    }

    /**
     * Encrypt any sensitive data.
     *
     * @param  string $key
     * @param  array  $formData
     * @param  array  $errors
     * @return array
     */
    protected function applyTableConfiguration($key, $formData, &$errors)
    {
        try {
            /** @var TableConfigService $tableConfigService */
            $tableConfigService = app(TableConfigService::class);
            $formData = $tableConfigService->apply([$key], $formData);
            $errors   = array_merge($errors, $tableConfigService->getErrors());
        } catch (\Exception $e) {
            logger()->alert("Unable to apply table config service to '$key'.");
        }

        return $formData;
    }
}
