<?php

namespace Ignite\Claim\Tests\Unit\Entities;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Ignite\Claim\Entities\Offer;
use Ignite\Claim\Entities\OfferPromotion;
use Ignite\Claim\Entities\OfferValue;
use Ignite\Claim\Entities\Rule;
use Ignite\Claim\Entities\RuleCalculate;
use Ignite\Claim\Entities\RuleCondition;
use Ignite\Claim\Entities\RuleOffer;
use Ignite\Claim\Entities\RuleParticipant;
use Ignite\Claim\Entities\RuleValue;
use Ignite\Core\Entities\Participant;
use Ignite\Tests\TestCase;

class RuleCalculateTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     * @group Rule
     */
    public function it_will_terminate_when_claim_data_is_not_passed()
    {
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams();
        unset($params['claim']);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertFalse($value->result);
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_terminate_when_a_claim_id_cannot_be_found_and_no_claim_data_is_passed()
    {
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams();
        unset($params['claim']);
        $params['claim_id'] = 1;
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertFalse($value->result);
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     * @expectedException \DomainException
     */
    public function it_will_terminate_when_an_offer_promotion_id_is_not_passed()
    {
        factory(Rule::class)->create(['status' => 1]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams();
        unset($params['claim']['offer_promotion_id']);

        $calculator->value($params);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_terminate_when_there_are_no_rules_to_process()
    {
        $promotion = factory(OfferPromotion::class)->create();
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams();
        $params['claim']['offer_promotion_id'] = $promotion->getKey();

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertFalse($value->result);
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_rules_are_passed_as_a_string_of_comma_delimited_ids()
    {
        $offer = factory(Offer::class)->create();
        $rule1 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);
        $rule2 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => $rule1->getKey() . ',' . $rule2->getKey()]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_rules_are_passed_as_a_string_of_semicolon_delimited_ids()
    {
        // Arrange
        $offer = factory(Offer::class)->create();
        $rule1 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);
        $rule2 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);

        // Act
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => $rule1->id . ';' . $rule2->id]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $value = $calculator->value($params);

        // Assert
        $actual = $value->data->values['claim_total'];
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_rules_are_passed_as_a_string_of_newline_delimited_ids()
    {
        $offer = factory(Offer::class)->create();
        $rule1 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);
        $rule2 = factory(Rule::class)->create(['offer_promotion_id' => $offer->promotion_id, 'status' => 1]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $rules = <<<RULES
{$rule1->id}
{$rule2->id}
RULES;
        $params = $this->getValueParams(['ruleIds' => $rules]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_the_rules_evaluate_to_false()
    {
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => 1]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_terminate_when_a_participant_type_is_not_passed()
    {
        $offer = factory(Offer::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);

        $params = $this->getValueParams();
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        unset($params['participantType']);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertFalse($value->result);
        $this->assertEquals('Unable to determine Participant Type.', $value->errors[0]);
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_a_participant_type_is_passed_in_via_participant_data_and_not_passed_directly()
    {
        $participant = factory(Participant::class)->create();
        $offer = factory(Offer::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        unset($params['participantType']);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_a_participant_type_is_passed_in_via_participant_data_with_a_guessable_key_and_not_passed_directly()
    {
        $offer = factory(Offer::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);

        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $this->newParticipant();
        unset($params['participantType']);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_rule_id_is_not_in_rule_list()
    {
        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => '520']);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $value = $calculator->value($params);

        $actual = $value->data->values['claim_total'];

        $this->assertContains("No matching Rules in list (520), for Promotion Id = '{$offer->promotion_id}'.", $value->errors);
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_only_inactive_rules_are_available()
    {
        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 0
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_only_basic_checks_fail()
    {
        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 0
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_the_correct_total_for_multiple_lineitems_when_using_the_max_per_claim_limit()
    {
        $promotion = factory(OfferPromotion::class)->create();
        $offer1 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 1',
            'value' => 100
        ]);
        $offer2 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 2',
            'value' => 100
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $promotion->getKey(),
            'status' => 1,
            'max_times' => 1,
        ]);
        $participant = factory(Participant::class)->create();

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $promotion->getKey();
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'] = [];
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer1->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 2',
            'qty' => 2
        ]);
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer2->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 2',
            'qty' => 3
        ]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        // The first offer has a quantity of 2 and
        // the rule should only run once, so expect the value to be 200
        $expected = $offer1->value * 2;

        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_the_claim_created_at_date_is_before_the_rule_start_date()
    {
        $now = now();
        $start = now()->endOfMonth();

        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            "start_date" => $start,
            "end_date" => null,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['created_at'] = $now->format('Y-m-d H:i:s');
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 2
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $now = $now->format('m/d/Y');
        $start = $start->format('m/d/Y');

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry(
            $value->data->logs,
            "Rule Id {$rule->getKey()}: 'Default', Basic Checks: Start Date '{$now}' >= '{$start}', Failed."
        );
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_the_claim_created_at_date_is_after_the_rule_end_date()
    {
        $now = now();
        $end = now()->subDay();

        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            "start_date" => null,
            "end_date" => $end->format('Y-m-d H:i:s'),
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['created_at'] = $now->format('Y-m-d H:i:s');
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 2
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $now = $now->format('m/d/Y');
        $end = $end->format('m/d/Y');

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry(
            $value->data->logs,
            "Rule Id {$rule->getKey()}: 'Default', Basic Checks: End Date '{$now}' < '{$end}', Failed."
        );
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_active_rules_are_outside_the_rule_date()
    {
        $now = now();
        $start = now()->endOfMonth();
        $end = now()->addMonth()->endOfMonth();

        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            "start_date" => $start,
            "end_date" => $end,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['created_at'] = $now->format('Y-m-d H:i:s');
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 2
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $now = $now->format('m/d/Y');
        $start = $start->format('m/d/Y');

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry(
            $value->data->logs,
            "Rule Id {$rule->getKey()}: 'Default', Basic Checks: Start Date '{$now}' >= '{$start}', Failed."
        );
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_return_zero_value_when_no_lineitems_exist()
    {
        $offer = factory(Offer::class)->create(['value' => 100]);
        $participant = factory(Participant::class)->create();
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_with_a_fixed_value()
    {
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $participant = factory(Participant::class)->create();

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offer->value, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_with_multiple_line_items_with_a_fixed_value()
    {
        $participant = factory(Participant::class)->create();
        $promotion = factory(OfferPromotion::class)->create();
        $offer1 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 1',
            'value' => 100
        ]);
        $offer2 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 2',
            'value' => 100
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $promotion->getKey(),
            'status' => 1
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $promotion->getKey();
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'] = [];
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer1->getKey(),
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 2
        ]);
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer1->getKey(),
            'offer_group' => 'Test Group 1',
            'name' => 'Test 2',
            'qty' => 3
        ]);


        $value = $calculator->value($params);

        $expected = ($offer1->value * 2) + ($offer2->value * 3);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_with_a_percent_value()
    {
        $saleValue = 1000;
        $offer = factory(Offer::class)->create([
            'value_type_id' => Offer::VALUE_TYPE_PERCENT,
            'value' => 50
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $participant = factory(Participant::class)->create();

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['sale_value'] = $saleValue;
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($saleValue / 2, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_with_multiple_line_items_with_a_percent_value()
    {
        $saleValue = 1000;
        $participant = factory(Participant::class)->create();
        $promotion = factory(OfferPromotion::class)->create();
        $offer1 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 1',
            'value_type_id' => Offer::VALUE_TYPE_PERCENT,
            'value' => 25
        ]);
        $offer2 = factory(Offer::class)->create([
            'promotion_id' => $promotion->getKey(),
            'name' => 'Test 2',
            'value_type_id' => Offer::VALUE_TYPE_PERCENT,
            'value' => 10
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $promotion->getKey(),
            'status' => 1
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $promotion->getKey();
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['sale_value'] = $saleValue;
        $params['claim']['lineitems'] = [];
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer1->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 2
        ]);
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer2->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 2',
            'qty' => 3
        ]);


        $value = $calculator->value($params);

        // 25% + 10% of 1000 = 350
        $expected = ($saleValue / 4) + ($saleValue / 10);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(
            $expected,
            $actual,
            '25% + 10% of 1000 = 350'
        );
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_for_only_specific_participant_type()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'participants_mode_id' => 2,
        ]);
        factory(RuleParticipant::class)->create([
            'rule_id' => $rule->id,
            'participant_type' => $participant->type,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(100, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_cannot_calculate_a_rule_based_on_the_current_offer_when_specific_participant_type_is_not_matched()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'participants_mode_id' => 2,
        ]);
        factory(RuleParticipant::class)->create([
            'rule_id' => $rule->id,
            'participant_type' => 'Account Executive',
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_for_only_specific_offer_type()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $notOffer = factory(Offer::class)->create(['value' => 1000]);
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'offers_mode_id' => Offer::VALUE_MODE_MULTIPLE,
        ]);
        factory(RuleOffer::class)->create([
            'rule_id' => $rule->id,
            'offer_id' => $offer->id,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $notOffer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(100, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_for_multiple_values_based_on_participant_type()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $offer = factory(Offer::class)->create([
            'value_mode_id' => Offer::VALUE_MODE_MULTIPLE,
            'value' => 100,
        ]);
        factory(OfferValue::class)->create([
            'offer_id' => $offer->id,
            'participant_type' => 'Account Executive',
            'value_type_id' => Offer::VALUE_TYPE_FIXED,
            'value' => 100,
        ]);
        $offerValue = factory(OfferValue::class)->create([
            'offer_id' => $offer->id,
            'participant_type' => $participant->type,
            'value_type_id' => Offer::VALUE_TYPE_FIXED,
            'value' => 75,
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'offers_mode_id' => Rule::OFFER_MODE_ALL,
        ]);
        factory(RuleOffer::class)->create([
            'rule_id' => $rule->id,
            'offer_id' => $offer->id,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1,
        ]);

        $value = $calculator->value($params);

        $expected = $offerValue->value;
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_can_calculate_a_rule_based_on_the_current_offer_for_multiple_values_based_on_participant_type_but_will_skip_if_the_participant_type_does_not_have_a_value()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $offer = factory(Offer::class)->create([
            'value' => 100,
            'value_mode_id' => Offer::VALUE_MODE_MULTIPLE
        ]);
        factory(OfferValue::class)->create([
            'offer_id' => $offer->id,
            'participant_type' => 'Account Executive',
            'value_type_id' => Offer::VALUE_TYPE_FIXED,
            'value' => 100
        ]);
        $offerValue = factory(OfferValue::class)->create([
            'offer_id' => $offer->id,
            'participant_type' => $participant->type,
            'value_type_id' => Offer::VALUE_TYPE_FIXED,
            'value' => 75
        ]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'offers_mode_id' => Rule::OFFER_MODE_ALL,
        ]);
        factory(RuleOffer::class)->create([
            'rule_id' => $rule->id,
            'offer_id' => $offer->id,
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = 'Foo';
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);

        $expected = 0; // No value, since the participant type `foo` has no offer value.
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_skip_conditions_when_the_comparison_table_doesnt_exist()
    {
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant']['participant_type'] = "Sales Manager";

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'fake_table|fake_field',
            'directive' => '=',
            'data_1' => "Sales",
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry(
            $value->data->logs,
            "Rule Id {$rule->getKey()}: 'Default', Conditions Checks Failed."
        );
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_on_the_claim_table()
    {
        $participant = factory(Participant::class)->create(['type' => 'Sales Manager']);
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
        ]);
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => '=',
            'data_1' => 500
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = 500;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(100, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_on_the_participant_table()
    {
        $participant = factory(Participant::class)->create([
            'type' => 'Sales Manager',
            'country' => 'US'
        ]);
        $offer = factory(Offer::class)->create(['value' => 100]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
        ]);
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|country',
            'directive' => '=',
            'data_1' => $participant->country
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = 500;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals(100, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_is_not_equal()
    {
        $offerValue = 100;

        $participant = factory(Participant::class)->create([
            'type' => 'Sales Manager',
            'country' => 'Ireland'
        ]);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|country',
            'directive' => '!=',
            'data_1' => 'US'
        ]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['participantType'] = $participant->type;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_is_greater_than()
    {
        $saleValue = 500;
        $conditionValue = 499;
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => '>',
            'data_1' => $conditionValue
        ]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = $saleValue;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_is_greater_than_or_equal()
    {
        $saleValue = 500;
        $conditionValue = 500;
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => '>=',
            'data_1' => $conditionValue
        ]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = $saleValue;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_is_less_than()
    {
        $saleValue = 499;
        $conditionValue = 500;
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => '<',
            'data_1' => $conditionValue
        ]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = $saleValue;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_is_less_than_or_equal()
    {
        $saleValue = 500;
        $conditionValue = 501;
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => '<=',
            'data_1' => $conditionValue
        ]);
        $params['claim']['sale_value'] = $saleValue;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;

        $calculator = app(RuleCalculate::class, ['debug' => null]);

        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_a_condition_is_between_two_values()
    {
        $saleValue = 750;
        $conditionValue1 = 500;
        $conditionValue2 = 800;
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|sale_value',
            'directive' => 'BETWEEN',
            'data_1' => $conditionValue1,
            'data_2' => $conditionValue2,
        ]);
        $params['claim']['sale_value'] = $saleValue;
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_a_condition_is_between_two_dates()
    {
        $createdAt = now()->format('Y-m-d H:i:s');
        $conditionValue1 = now()->subMonth(1);
        $conditionValue2 = now()->addMonth(1);
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|created_at',
            'directive' => 'BETWEEN',
            'data_1' => $conditionValue1,
            'data_2' => $conditionValue2,
        ]);
        $params['claim']['participant']['created_at'] = $createdAt;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_exists_in_list()
    {
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;

        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|country',
            'directive' => 'IN',
            'data_1' => "US\r\nIE\r\nDE",
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     * @group RuleCondition
     */
    public function it_will_check_a_condition_does_not_exist_in_list()
    {
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant']['participant_type'] = "Account Executive";
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|participant_type',
            'directive' => 'NOT IN',
            'data_1' => "Foo\r\nBar\r\nBaz",
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_a_condition_is_like_or_similar_to_the_provided_text()
    {
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant']['participant_type'] = "Sales Manager";
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'participant|participant_type',
            'directive' => 'LIKE',
            'data_1' => "Sales",
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $this->assertTrue($value->result);

        $actual = $value->data->values['claim_total'];

        $this->assertEquals($offerValue, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_continue_when_a_condition_directive_cannot_be_found()
    {
        $offerValue = 100;

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $params = $this->getBasicClaimParams($offer, $rule);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant']['participant_type'] = "Sales Manager";
        factory(RuleCondition::class)->create([
            'rule_id' => $rule->getKey(),
            'db_column' => 'claim|participant_type',
            'directive' => '',
            'data_1' => "Sales",
        ]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry(
            $value->data->logs,
            "Rule Id {$rule->getKey()}: 'Default', Conditions Checks Failed."
        );
        $this->assertEquals(0, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_participant_data_in_single_value_mode()
    {
        $offerValue = 100;

        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 2,
            'value' => 50
        ]);

        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($rule->value, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_participant_data_for_a_fixed_amount_value_specific_to_them()
    {
        $offerValue = 100;

        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 3
        ]);
        $ruleValue = factory(RuleValue::class)->create([
            'rule_id' => $rule->id,
            'participant_type' => $participant->type,
            'value_type_id' => 1,
            'value' => 50
        ]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = $ruleValue->value;
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * When calculating a percentage where the basis is not derived from the running balance of previous rules, the form
     * must have the `sale_value` field attached.
     *
     * @test
     * @group Rule
     */
    public function it_will_check_participant_data_for_a_percentage_amount_value_specific_to_them()
    {
        $saleValue = 1000;
        $offerValue = 100;

        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => Rule::VALUE_SOURCE_RULE_BY_PARTICIPANT
        ]);
        $ruleValue = factory(RuleValue::class)->create([
            'rule_id' => $rule->id,
            'participant_type' => $participant->type,
            'value_type_id' => Rule::VALUE_TYPE_PERCENT,
            'value' => 50
        ]);
        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);
        $params['claim']['sale_value'] = $saleValue;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = $saleValue * ($ruleValue->value / 100);
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_participant_data_in_single_value_mode_with_running_balance()
    {
        $offerValue = 100;

        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule1 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 2,
            'value' => 50
        ]);
        $rule2 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 4,
            'value' => 150
        ]);

        $params = $this->getValueParams(['ruleIds' => [$rule1->id, $rule2->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = $rule1->value + $rule2->value;
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertEquals($expected, $actual);
    }

    /**
     * @test
     * @group Rule
     */
    public function it_will_check_participant_data_for_fixed_amount_value_specific_to_the_claim_participant_with_running_balance()
    {
        $offerValue = 100;

        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule1 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 5,
        ]);
        $ruleValue1 = factory(RuleValue::class)->create([
            'rule_id' => $rule1->id,
            'participant_type' => $participant->type,
            'value_type_id' => 1,
            'value' => 50
        ]);
        $rule2 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 5
        ]);
        $ruleValue2 = factory(RuleValue::class)->create([
            'rule_id' => $rule2->id,
            'participant_type' => $participant->type,
            'value_type_id' => 1,
            'value' => 50
        ]);
        $params = $this->getValueParams(['ruleIds' => [$rule1->id, $rule2->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = $ruleValue1->value + $ruleValue2->value;
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * The running balance rule by participant is meant to be processed as the last rule in the rule list.
     * The general spirit of the rule is, once all rules are calculated, if you are Bob, you get x% of the total.
     * Typically, the use case scenario is, double all of Bob's points after all rules are calculated.
     *
     * @test
     * @group Rule
     * @group RuleLog
     */
    public function it_will_check_participant_data_for_percentage_amount_value_specific_to_the_claim_participant_with_running_balance()
    {
        $offerValue = 100;
        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule1 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 1,
        ]);
        $rule2 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 5
        ]);
        $ruleValue2 = factory(RuleValue::class)->create([
            'rule_id' => $rule2->id,
            'participant_type' => $participant->type,
            'value_type_id' => 2,
            'value' => 100 // 100%
        ]);
        $params = $this->getValueParams(['ruleIds' => [$rule1->id, $rule2->id]]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = $offerValue + ($offerValue * ($ruleValue2->value / 100));
        $actual = $value->data->values['claim_total'];

        $this->assertEquals($expected, $actual);
    }

    /**
     * Test when no value source or unknown value source is passed
     *
     * @test
     * @group Rule
     * @group RuleLog
     */
    public function it_will_create_a_default_configuration_for_value_source_when_value_mode_id_is_unknown_and_log_an_error()
    {
        $offerValue = 100;
        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => 0,
        ]);
        $params = $this->getValueParams();
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = 0;
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry($value->data->logs, sprintf(
            "ERROR: Rule Id %s: '%s', Source '%s'.",
            $rule->id,
            $rule->name,
            'Unknown Source Type'
        ));
        $this->assertEquals($expected, $actual);
    }

    /**
     * @todo Test for zero basis when value type is percent
     *
     * @test
     * @group Rule
     * @group RuleLog
     */
    public function it_will_log_a_warning_when_value_type_is_a_percentage_and_basis_is_zero()
    {
        $offerValue = 100;
        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);
        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1,
            'value_mode_id' => Rule::VALUE_SOURCE_RULE_SINGLE,
            'value_type_id' => Rule::VALUE_TYPE_PERCENT,
            'value' => 50
        ]);
        $params = $this->getValueParams();
        $lineitem = $this->getLineItem(['offer_id' => $offer->id]);
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = 0;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $lineitem;

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $expected = 0;
        $actual = $value->data->values['claim_total'];

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry($value->data->logs, sprintf(
            "WARNING: Rule Id %s: '%s', Line Item %s: '%s', %s basis is Zero.",
            $rule->id,
            $rule->name,
            $lineitem['id'],
            $lineitem['name'],
            'claim.sale_value'
        ));
        $this->assertEquals($expected, $actual);
    }

    /**
     * Test correct log exists when rule is disabled and skipped
     *
     * @test
     * @group Rule
     * @group RuleLog
     */
    public function it_will_log_a_message_when_a_rule_is_disabled_and_skipped()
    {
        $offerValue = 100;

        $user = $this->loginAsAdmin();
        $participant = factory(Participant::class)->create(['user_id' => $user->getKey()]);

        $offer = factory(Offer::class)->create(['value' => $offerValue]);
        $rule1 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 1
        ]);
        $rule2 = factory(Rule::class)->create([
            'offer_promotion_id' => $offer->promotion_id,
            'status' => 0
        ]);

        $params = $this->getValueParams();
        $params['claim']['offer_promotion_id'] = $offer->promotion_id;
        $params['claim']['sale_value'] = 0;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => $offer->id]);

        $calculator = app(RuleCalculate::class, ['debug' => null]);
        $value = $calculator->value($params);

        $this->assertTrue($value->result);
        $this->assertLogContainsEntry($value->data->logs, sprintf(
            "Rule Id %s: '%s', Disabled in Basic Checks, skipped...",
            $rule2->id,
            $rule2->name
        ));

        $expected = $offerValue;
        $actual = $value->data->values['claim_total'];
        $this->assertEquals($expected, $actual);
    }

    /**
     * Test that getRules() returns early with a response result of false when there are no rules for the given promotion.
     *
     * @test
     * @group Rule
     */
    public function it_returns_early_with_an_error_message_when_there_are_no_rules_for_the_given_promotion()
    {
        $promotion = factory(OfferPromotion::class)->create();

        $params = $this->getValueParams();
        $params['claim']['offer_promotion_id'] = $promotion->getKey();
        $params['claim']['lineitems'][] = $this->getLineItem(['offer_id' => 10]);
        $calculator = app(RuleCalculate::class, ['debug' => null]);

        $value = $calculator->value($params);

        $this->assertContains(
            sprintf("No Rules for Promotion Id = '%s'.", $promotion->getKey()),
            $value->errors
        );
    }

    /**
     * Get the value params for a claim rule calculation
     * @param array $overrides
     *
     * @return array
     */
    private function getValueParams(array $overrides = [])
    {
        return array_merge_recursive([
            // Passed via the Form
            'claim' => [
                'claim_id' => 'new',
                'id' => 'new',
                'mode' => 'new',
                'sale_value' => 0,
                'created_at' => now(),
                'offer_promotion_id' => 1,
                //'maxParticipants' => 0,
                //'maxLineItems' => 5,
                'participant' => [],
                'lineitems' => []
            ],
            // Usually Passed via the Controller
            'activeOnly' => 1,
            'participantType' => 'Account Executive',
            'debugMode' => 0,
            'debugLevel' => 10,
            'detailLevel' => 10,
        ], $overrides);
    }

    /**
     * Create a new participant.
     *
     * @param  array $overrides
     * @return array
     */
    private function newParticipant(array $overrides = [])
    {
        return array_merge([
            'email' => 'foo@gmail.com',
            'participant_type' => 'Account Executive',
        ], $overrides);
    }

    private function getLineItem(array $overrides = [])
    {
        return array_merge([
            'id' => 'new',
            'offer_id' => 1,
            'offer_group' => 'Test Group 1',
            'name' => 'Test',
            'qty' => 1,
            'class' => 'claim_lineitem',
            'claim_id' => 1,
            'status' => 1,
        ], $overrides);
    }

    /**
     * Get the basic parameters required to calculate a rule for the claim participant.
     *
     * @param  Offer $offer
     * @param  Rule  $rule
     * @param  array $overrides
     * @return array
     */
    private function getBasicClaimParams($offer, $rule, array $overrides = [])
    {
        $participant = factory(Participant::class)->create(['type' => 'Account Executive']);

        $params = $this->getValueParams(['ruleIds' => [$rule->id]]);
        $params['participantType'] = $participant->type;
        $params['claim']['participant'] = $participant->toArray();
        $params['claim']['lineitems'][] = $this->getLineItem([
            'offer_id' => $offer->id,
            'offer_group' => 'Test Group 1',
            'name' => 'Test 1',
            'qty' => 1
        ]);

        return array_merge($params, $overrides);
    }

    public function setUp()
    {
        parent::setUp();

        $this->loadHelpers();

        /** @var \Ignite\Claim\Database\Seeders\TableTableSeeder $tableSeeder */
        $tableSeeder = app(\Ignite\Claim\Database\Seeders\TableTableSeeder::class);
        $tableSeeder->run();

        /** @var \Ignite\Claim\Database\Seeders\FormTableSeeder $tableSeeder */
        $formSeeder = app(\Ignite\Claim\Database\Seeders\FormTableSeeder::class);
        $formSeeder->run();
    }

    /**
     * Determine whether the array contains the message.
     *
     * @param  array  $logs
     * @param  string $message
     * @return bool
     */
    private function assertLogContainsEntry($logs, $entry, $message = '')
    {
        $condition = (bool) collect($logs)
            ->filter(function ($log) use ($entry) {
                return $log['message'] === $entry;
            })
            ->count();

        static::assertThat($condition, static::isTrue(), $message);

        return $condition;
    }
}
