<?php

namespace Ignite\Claim\Models\Summary;

use Ignite\Core\Facades\Format;

class Metric
{
    /**
     * @var string
     */
    protected $name;

    /**
     * @var float
     */
    protected $total;

    /**
     * @var int
     */
    protected $count;

    /**
     * @var array
     */
    protected $pending = [
        'class' => 'orange',
        'color' => '#FB9026',
    ];

    /**
     * @var array
     */
    protected $declined = [
        'class' => 'red',
        'color' => '#FF3949',
    ];

    /**
     * @var array
     */
    protected $approved = [
        'class' => 'green',
        'color' => '#5BD130',
    ];

    /**
     * @var array
     */
    protected $issued = [
        'class' => 'blue',
        'color' => '#64B5F6',
    ];

    /**
     * Metric constructor.
     *
     * @param string $name
     * @param float $total
     * @param int $count
     */
    public function __construct($name, $total, $count)
    {
        $this->name = $name;
        $this->total = $total;
        $this->count = $count;
    }

    /**
     * The name of the metric.
     *
     * @return string
     */
    public function name()
    {
        return $this->name;
    }

    /**
     * The count of the metric.
     *
     * @return mixed
     */
    public function count()
    {
        return Format::balance($this->count);
    }

    /**
     * The total value/amount of the metric.
     *
     * @return string
     */
    public function total()
    {
        return Format::balance($this->total);
    }

    /**
     * The percentage of the given total.
     *
     * @param float|int $total
     * @return float|int
     */
    public function percentage($total)
    {
        if ($total <= 0) {
            return 0;
        }

        return $this->count / $total;
    }

    /**
     * Get the configured color name for the metric.
     *
     * @param  string $default
     * @return string
     */
    public function color($default = '#dddddd')
    {
        return $this->statusOrDefault('color', $default);
    }

    /**
     * Get the configured class name for the metric.
     *
     * @param  string $default
     * @return string
     */
    public function class($default = 'default')
    {
        return $this->statusOrDefault('class', $default);
    }

    /**
     * Try to get the type of data from the status or provide a default.
     *
     * @param  $type
     * @param  $default
     * @return string
     */
    protected function statusOrDefault($type, $default)
    {
        $status = $this->name;

        if (! property_exists($this, $status)) {
            return $default;
        }

        return $this->$status[$type];
    }
}