<?php

namespace Ignite\Claim\Repositories;

use Illuminate\Support\Facades\DB;
use Ignite\Claim\Traits\ServiceProviderCommon;
use Ignite\Claim\Entities\Exception;

class DataRepository
{
    use ServiceProviderCommon;

    protected $class        = false;
    protected $classPath    = false;
    protected $object       = false;
    protected $processed    = false;
    protected $lastParams   = false;
    protected $response     = false;
    protected $dataFields       = [];
    protected $dataFieldsLoaded = false;
    protected $filters          = [];
    protected $dbColumns        = [];
    protected $dynamicColumns   = [];
    protected $matchingColumns  = [];

    /**
     * DataRepository constructor.
     *
     * @param  string $classPath
     * @param  array  $filters
     * @throws Exception
     */
    public function __construct($classPath, $filters = [])
    {
        $this->classPath    = $classPath;
        $this->class        = basename($this->classPath);
        $this->filters      = $filters;
        $this->object       = new $this->classPath();

        // --------------------------------------
        // Get Data Fields for this Class
        // --------------------------------------
        // Default to None
        $this->dataFields   = [];

        // --------------------------------------
        // Note: This did not do the right thing, in that it just got the
        //       class::$fields, which would not likely be correct at this point
        //       in time, since they have not been loaded from the Settings yet.
        //       This has be supplanted by "Just In Time" logic in the
        //       DynamicModel class.  see:  __get(), __set()
        // --------------------------------------
        // Check if there are Fields Defined in the Settings
        if (! empty($this->object->isDynamicModel)) {
            /*
            $dataFields = $this->object->getDataFields();
            if ( ! empty($dataFields) ) {
                $this->dataFields   = $dataFields;
            }
            */
            // --------------------------------------
            // Do Nothing here, because dataFields will be populated "Just In Time"
            // by the DynamicColumn class.  see: __get(), __set()
            // This way we won't have any dataFields defined, and we won't fail silently
            // if there are no Setting Columns for this Table.
            // --------------------------------------
        } elseif (method_exists($this->object, 'getFields')) {
            // Check if Class has ValidatesFields Traits
            $this->dataFields = $this->object->getFields();
            $this->setDefaultValues();
        } elseif (isset(($this->classPath)::$fields)) {
            // Check if Class has fields defined
            $this->dataFields = ($this->classPath)::$fields;
            $this->setDefaultValues();
        }

        // Handle non-DynamicModels that are missing their $fields member
        if (empty($this->dataFields) && empty($this->object->isDynamicModel)) {
            $message  = sprintf(
                "DataProvider: Class '%s' => '%s' has no Data Fields defined, can't contine.\n",
                $this->class,
                $this->classPath
            );
            $message .= sprintf("Please add the static::\$fields member to the Class.\n");
            throw new Exception($message);
        }
    }

    /**
     * Get the data field for the dynamic model.
     *
     * @return array
     */
    private function getDataFields()
    {
        if (! empty($this->dataFields)) {
            return $this->dataFields;
        }

        // -------------------------------
        // Handle "Just In Time" getting of dataFields for DynamicModel's
        // -------------------------------
        if (! empty($this->object->isDynamicModel)) {
            $dataFields = $this->object->getDataFields();
            if (! empty($dataFields)) {
                $this->dataFields   = $dataFields;
                $this->setDefaultValues();
            }
        }

        return $this->dataFields;
    }

    /**
     * Get a dynamic property.
     *
     * @param  string $name
     * @return bool|string
     */
    public function __get($name)
    {
        if (array_key_exists($name, $this->getDataFields())) {
            return $this->object->$name;
        }

        $value = false;

        switch ($name) {
            case 'lastParams':
            case 'svc_lastParams':
                $value = $this->lastParams;
                break;

            case 'lastResult':
            case 'svc_lastResult':
                $value = $this->response->result;
                break;

            case 'errors':
            case 'svc_errors':
                $value = $this->response->errors;
                break;

            case 'lastRepsonse':
            case 'svc_lastRepsonse':
                $value = $this->response;
                break;

            case 'svc_class':
                $value = $this->class;
                break;

            case 'svc_classPath':
                $value = $this->classPath;
                break;

            case 'svc_dataFields':
                $value = $this->getDataFields();
                break;

            case 'svc_filters':
                $value = $this->filters;
                break;
        }

        return $value;
    }

    public function __set($name, $value)
    {
        if (array_key_exists($name, $this->getDataFields())) {
            $this->object->$name = $value;
            return true;
        }

        return false;
    }

    public function getLastResponse()
    {
        return $this->response;
    }

    public function getDefaultValues()
    {
        $defaultValues = [];

        foreach ($this->getDataFields() as $key => $settings) {
            if (array_key_exists('default', $settings)) {
                $defaultValues[$key] = $settings['default'];
            }
        }

        return $defaultValues;
    }

    public function setDefaultValues()
    {
        foreach ($this->getDataFields() as $key => $settings) {
            if (array_key_exists('default', $settings)) {
                $this->object->$key = $settings['default'];
            }
        }

        return $this->toArray();
    }

    public function checkDbColumnsChanged(&$errors)
    {
        $dbColumnsChanged = false;

        if (! empty($this->object->isDynamicModel)) {
            $dbColumnsChanged = $this->object->checkDbColumnsChanged($errors);
        }

        return $dbColumnsChanged;
    }

    protected function populate($data, $fields = [])
    {
        if (empty($fields)) {
            $fields = array_keys($this->getDataFields());
        }

        $dataFields = $this->getDataFields();

        foreach ($fields as $key) {
            if (array_key_exists($key, $data)) {
                $type = (! empty($dataFields[$key]['type'])) ? $dataFields[$key]['type'] : 'unknown';
                $this->object->$key = self::dbFormat($data[$key], $type);
            }
        }

        return $this;
    }

    public function data($data, $fields = [])
    {
        if (empty($fields)) {
            $fields = array_keys($this->getDataFields());
        }

        $fieldData = [];
        foreach ($fields as $key) {
            if (array_key_exists($key, $data)) {
                $fieldData[$key] = $data[$key];
            }
        }

        return $fieldData;
    }

    public function validate($data, $validateFields = false)
    {
        // ------------------------------------------
        // Check if we were given an Array of Data rows
        // If not, convert to an Array of Data rows
        // ------------------------------------------
        if (! self::isArray($data)) {
            $wasArray = false;
            $dataArray = [$data];
        } else {
            $wasArray   = true;
            $dataArray  = $data;
        }

        // ------------------------------------------
        // Process Individual Data rows
        // ------------------------------------------
        foreach ($dataArray as $index => $data) {
            $errors = $this->classPath::validate($data, $validateFields);
            if (! empty($errors)) {
                if ($wasArray == true) {
                    $message = sprintf(
                        "Validation Error: %s, item number %s.",
                        $this->class,
                        number_format($index + 1)
                    );
                    array_unshift($errors, $message);
                }
                break;
            }
        }

        return $errors;
    }

    protected function checkRequireds($function, $params, $requireds)
    {
        if (self::isArray($params)) {
            $checkParams = $params[0];
        } else {
            $checkParams = $params;
        }

        foreach ($requireds as $required) {
            if (! array_key_exists($required, $checkParams)) {
                $this->response->result     = false;
                $this->response->errors[]   = sprintf("%s: missing param [%s].", $function, $required);
            }
        }

        return $this->response->result;
    }

    protected function getPrimaryKeyName($function)
    {
        $primaryKeyName = $this->object->getKeyName();

        if ($primaryKeyName == '') {
            $this->response->errors[] = sprintf(
                "%s: WARNING: primary key not defined on class %s, defaulting to 'id'.",
                $function,
                $this->class
            );
            $primaryKeyName = 'id';
        }

        return $primaryKeyName;
    }

    /**
     * Get the primary key value.
     *
     * @param  string $function
     * @return bool
     */
    protected function getPrimaryKeyValue($function)
    {
        $primaryKeyName = $this->getPrimaryKeyName($function);

        if (isset($this->object->$primaryKeyName)) {
            return $this->object->$primaryKeyName;
        }

        return false;
    }

    /**
     * Find a record.
     *
     * @param  array $params
     * @return bool
     */
    protected function find($params)
    {
        $function = __FUNCTION__;

        $primaryKey = $this->getPrimaryKeyName($function);
        $requireds  = [$primaryKey];

        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        // -----------------------------
        // Intitialize the Query
        // -----------------------------
        $query = $this->classPath::query();

        if (! $query) {
            return false;
        }

        // -----------------------------
        // Add Optional Clauses
        // -----------------------------
        $query = $this->addOptionalClauses($function, $query, $params);

        // -----------------------------
        // Check if the Requester only wants the Query Object back
        // -----------------------------
        if (! empty($params['mode']) && $params['mode'] == 'query') {
            $this->response->data = $query;
            return $this->response->result;
        }

        // -----------------------------
        // Run the Query
        // -----------------------------
        if (! empty($params['withSql'])) {
            DB::enableQueryLog();
        }

        $this->response->data = $query->find($params[$primaryKey]);

        if (! empty($params['withSql'])) {
            $queryLogs = DB::getQueryLog();
            $this->response->sql[] = $queryLogs[0]['query'];
            DB::disableQueryLog();
        }

        // -----------------------------
        // Check for Errors
        // -----------------------------
        if (! $this->response->data) {
            $this->response->result     = false;
            $this->response->errors[]   = sprintf("%s: query failed.", $function);
        }

        return $this->response->result;
    }

    protected function addWhere($function, &$query, $fieldName, $fieldValue, $filterParams)
    {
        // -------------------------------
        // Check if Value has an Operator Override eg:  '!=blah'
        // Also, if there are Multiple '|' delimited values for the Operater, ie: between:100|500
        // -------------------------------
        list($operator, $value) = self::getOperatorValue($fieldValue, $filterParams);
        $values = explode('|', $value);

        // -------------------------------
        // Add correct type of where()
        // -------------------------------
        $type = (! empty($filterParams['type'])) ? $filterParams['type'] : 'none';

        foreach ($values as $index => $oneValue) {
            switch ($type) {
                case 'date':
                case 'datetime':
                    // $dateStamp = date('Y-m-d H:i:s', strtotime($value));
                    $dateStamp = $this->dbFormat($oneValue, $type);
                    if (! $dateStamp && $dateStamp !== null) {
                        $this->response->errors[] = sprintf(
                            "%s: Invalid date for %s: '%s'.",
                            $function,
                            $fieldName,
                            $oneValue
                        );
                        return false;
                    }
                    $values[$index] = $dateStamp;
                    break;

                case 'integer':
                case 'none':
                case 'string':
                case 'unknown':
                default:
                    break;
            }
        }

        switch ($operator) {
            case 'null':
            case 'isnull':
                $query->whereNull($fieldName);
                break;

            case 'notnull':
                $query->whereNotNull($fieldName);
                break;

            case 'in':
            case 'isin':
                $query->whereIn($fieldName, $values);
                break;

            case 'notin':
                $query->whereNotIn($fieldName, $values);
                break;

            case 'between':
                if (count($values) != 2) {
                    $this->response->errors[] = sprintf(
                        "%s: Invalid number of values %s, for BETWEEN on '%s'.",
                        $function,
                        $fieldName,
                        count($oneValue)
                    );
                    return false;
                }
                $query->whereBetween($fieldName, $values);
                break;

            default:
                $query->where($fieldName, $operator, $value);
                break;
        }

        return true;
    }

    protected function addQueryFilters($function, &$query, $filters, $queryParams)
    {
        $rc = true;

        $matchingColumns = $this->getMatchingColumns('filter', $filters);

        // -------------------------------
        // Have to Add each Where separately
        // Also, can only use actual Database Columns when building Query
        // -------------------------------
        foreach ($matchingColumns['dbColumns'] as $fieldName => $filterParams) {
            if (! isset($queryParams[$fieldName])) {
                continue;
            }

            $rc = $this->addWhere($function, $query, $fieldName, $queryParams[$fieldName], $filterParams);
            if (! $rc) {
                break;
            }
        }

        return $rc;
    }

    protected function addSelect($function, &$query, $queryParams)
    {
        $select = false;

        if (isset($queryParams['select'])) {
            $select = $queryParams['select'];
        }

        if (isset($queryParams['SELECT'])) {
            $select = $queryParams['SELECT'];
        }

        if ($select) {
            if (! is_array($select)) {
                $select = preg_split('/ *, */', $select);
            }

            // ----------------------------
            // Get and Add the matching Columns to our Columns Lookup list
            // We may need it later and/or for Post SQL processing
            // ----------------------------
            $matchingColumns = $this->getMatchingColumns('select', $select);

            // ----------------------------
            // Can only use actual Database Columns when building Query
            // ----------------------------
            $selectColumns = $matchingColumns['dbColumns'];

            // But, if this Model has Dynamic Columns, we must retrieve the Dynamic Column as well
            if (! empty($this->object->isDynamicModel) && ! empty($matchingColumns['dynamicColumns'])) {
                $dynamicColumnName = $this->object->getDynamicColumnName();
                if (! in_array($dynamicColumnName, $selectColumns)) {
                    $selectColumns[] = $dynamicColumnName;
                }
            }

            $query->select($selectColumns);
        }

        return $query;
    }

    protected function addDistinct($function, &$query, $queryParams)
    {
        $distinct = false;

        if (isset($queryParams['distinct'])) {
            $distinct = $queryParams['distinct'];
        }

        if (isset($queryParams['DISTINCT'])) {
            $distinct = $queryParams['DISTINCT'];
        }

        if ($distinct) {
            if (! is_array($distinct)) {
                $distinct = preg_split('/ *, */', $distinct);
            }

            // ----------------------------
            // Get and Add the matching Columns to our Columns Lookup list
            // We may need it later and/or for Post SQL processing
            // ----------------------------
            $matchingColumns = $this->getMatchingColumns('distinct', $distinct);

            // ----------------------------
            // Can only use actual Database Columns when building Query
            // ----------------------------
            $selectColumns = $matchingColumns['dbColumns'];

            // ---------------------------------------------------------------------------
            // Need to be able to do a DISTINCT on Dynamic Columns at some point, but can't yet
            // ---------------------------------------------------------------------------
            // But, if this Model has Dynamic Columns, we must retrieve the Dynamic Column as well
            /*
                        if ( ! empty($this->object->isDynamicModel) && ! empty($matchingColumns['dynamicColumns']) ) {
                            $dynamicColumnName = $this->object->getDynamicColumnName();
                            if ( ! in_array($dynamicColumnName, $selectColumns) ) {
                                $selectColumns[] = $dynamicColumnName;
                            }
                        }
            */

            $query->distinct($selectColumns);
        }

        return $query;
    }

    protected function addOrderBy($function, &$query, $queryParams)
    {
        $orderBys = false;

        if (isset($queryParams['orderby'])) {
            $orderBys = $queryParams['orderby'];
        }

        if (isset($queryParams['ORDERBY'])) {
            $orderBys = $queryParams['ORDERBY'];
        }

        if (isset($queryParams['OrderBy'])) {
            $orderBys = $queryParams['OrderBy'];
        }

        if ($orderBys) {
            if (! is_array($orderBys)) {
                $orderBys = preg_split('/ *, */', $orderBys);
            }
            $matchingColumns = $this->getMatchingColumns('orderby', $orderBys);

            // -------------------------------
            // Have to Add each Order By separately
            // Also, can only use actual Database Columns when building Query
            // -------------------------------
            foreach ($matchingColumns['dbColumns'] as $orderBy) {
                // Check if ASC or DESC is on the Order By
                $parts = explode('|', $orderBy);
                if (count($parts) > 1) {
                    $query->orderBy($parts[0], $parts[1]);
                } else {
                    $query->orderBy($parts[0]);
                }
            }
        }

        return $query;
    }

    protected function addOptionalClauses($function, &$query, $params)
    {
        // -----------------------------
        // Add any Child / Related records requested
        // -----------------------------
        if (! empty($params['with'])) {
            $query = $query->with($params['with']);
        }

        // -----------------------------
        // Add any Field Select requested
        // -----------------------------
        $query = $this->addSelect($function, $query, $params);

        // -----------------------------
        // Add any Distinct Select requested
        // -----------------------------
        $query = $this->addDistinct($function, $query, $params);

        // -----------------------------
        // Add Query Filters
        // -----------------------------
        $rc = $this->addQueryFilters($function, $query, $this->filters, $params);
        if (! $rc) {
            $this->response->result = false;
            return false;
        }

        // -----------------------------
        // Add any Order By's requested
        // -----------------------------
        $query = $this->addOrderBy($function, $query, $params);

        // -----------------------------
        // Add any Limit requested
        // -----------------------------
        if (! empty($params['limit'])) {
            $query = $query->limit($params['limit']);
        }

        return $query;
    }

    protected function getList($params)
    {
        $function = __FUNCTION__;

        // -----------------------------
        // Intitialize the Query
        // -----------------------------
        $query = $this->classPath::query();

        // -----------------------------
        // Add Optional Clauses
        // -----------------------------
        $query = $this->addOptionalClauses($function, $query, $params);
        if (! $query) {
            return false;
        }

        // -----------------------------
        // Check if the Requester only wants the Query Object back
        // -----------------------------
        if (! empty($params['mode']) && $params['mode'] == 'query') {
            $this->response->data = $query;
            return $this->response->result;
        }

        // -----------------------------
        // Run the Query
        // -----------------------------
        if (! empty($params['withSql'])) {
            DB::enableQueryLog();
        }

        $this->response->data = $query->get();

        if (! empty($params['withSql'])) {
            $queryLogs = DB::getQueryLog();
            $this->response->sql[] = $queryLogs[0]['query'];
            DB::disableQueryLog();
        }

        // -----------------------------
        // Check for Errors
        // -----------------------------
        if (! $this->response->data) {
            $this->response->result = false;
            $this->response->errors[] = sprintf("%s: query failed.", $function);
        }

        return $this->response->result;
    }

    protected function addDynamicColumns($dataObject, $dataArray)
    {
        // Handle case where Dynamic Columns were in a Select
        if (! empty($this->matchingColumns['select'])) {
            if (! empty($this->matchingColumns['select']['dynamicColumns'])) {
                foreach ($this->matchingColumns['select']['dynamicColumns'] as $name) {
                    $dataArray[$name] = $dataObject->$name;
                }
            }
        } else {
            // If no Select, return All Dynamic Columns
            if (! empty($this->dynamicColumns)) {
                foreach ($this->dynamicColumns as $name => $stuff) {
                    $dataArray[$name] = $dataObject->$name;
                }
            }
        }

        return $dataArray;
    }

    /**
     * Run post-processing on the
     * @param  array $responseObjects
     * @param  array $dataArray
     * @return array|mixed
     */
    protected function postProcessArrayData($responseObjects, $dataArray)
    {
        // Add Dynamic Fields to response data if it is an Array
        if (! isset($responseObjects[0])) {
            $modifiedData = $this->addDynamicColumns($responseObjects, $dataArray);
        } else {
            $modifiedData = [];
            foreach ($responseObjects as $index => $responseObject) {
                $modifiedData[$index] = $this->addDynamicColumns($responseObject, $dataArray[$index]);
            }
        }

        return $modifiedData;
    }

    /**
     * Build up a response object.
     *
     * @param  array $params
     * @return bool
     */
    protected function buildResponse($params)
    {
        if ($this->processed) {
            return $this->response;
        }

        $mode = (! empty($params['mode'])) ? $params['mode'] : 'default';

        if (! empty($this->response->data)) {
            switch ($mode) {
                case 'array':
                    if (! is_array($this->response->data) && ! method_exists($this->response->data, 'toArray')) {
                        $dataArray = (array) $this->response->data;
                    } else {
                        $dataArray = $this->response->data->toArray();
                    }

                    // ----------------------------
                    // Do any Post Processing needed for Dynamic Columns
                    // ----------------------------
                    if (! empty($dataArray)) {
                        $this->response->data = $this->postProcessArrayData($this->response->data, $dataArray);
                    } else {
                        $this->response->data = $dataArray;
                    }

                    break;

                case 'json':
                    $this->response->data = json_encode($this->response->data);
                    break;

                case 'query':
                    // Do nothing, find() and getList() will have already populated with a Query Object
                    break;

                case 'object':
                case 'default':
                default:
                    // Do nothing, already in this format
                    break;
            }
        }

        $this->processed = true;

        return $this->response;
    }

    private function setStaticAndDynamicColumns()
    {
        if (empty($this->dbColumns)) {
            // Build Separate lists of DB and Dynamic Fields
            if (! empty($this->object->isDynamicModel)) {
                foreach ($this->getDataFields() as $name => $dataField) {
                    if ($this->object->isDynamicColumn($name)) {
                        $this->dynamicColumns[$name] = true;
                    } else {
                        $this->dbColumns[$name] = true;
                    }
                }
            } else {
                // Not a Dynamic Model, so ALL Fields must be DB Fields
                $this->dbColumns = $this->getDataFields();
            }
        }

        if (! empty($this->dynamicColumns)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Get columns matching by key.
     *
     * @param  string $key
     * @param  array $columns
     * @return mixed
     */
    private function getMatchingColumns($key, $columns)
    {
        if (! empty($this->matchingColumns[$key])) {
            return $this->matchingColumns[$key];
        }

        // ------------------------------
        // Check if this is just a List of Names, or a set of Name => Value pairs
        // ------------------------------
        $isIndexArray = (! empty($columns[0])) ? true : false;

        // ------------------------------
        // Build separate lists of Database and Dynamic Columns by Use Key
        // ------------------------------
        $matchingColumns = [];
        $matchingColumns ['dbColumns']         = [];
        $matchingColumns ['dynamicColumns']    = [];

        foreach ($columns as $index => $value) {
            $name = ($isIndexArray) ? $value : $index;
            if (! empty($this->dynamicColumns[$name])) {
                $matchingColumns['dynamicColumns'][$name]  = $name;
            } else {
                $matchingColumns['dbColumns'][$name]       = $name;
            }
        }

        // ------------------------------
        // Save Database and Dynamic Columns by Use Key for Post Query Result processing
        // ie: filtering, sorting, etc...
        // ------------------------------
        $this->matchingColumns[$key] = $matchingColumns;

        return $this->matchingColumns[$key];
    }

    public function get($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $requireds = ['action'];
        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        $this->setStaticAndDynamicColumns();

        $action = $params['action'];

        switch ($action) {
            case 'find':
                $rc = $this->find($params);
                break;

            case 'getList':
                $rc = $this->getList($params);
                break;

            default:
                $this->response->result = false;
                $this->response->errors[] = sprintf(
                    "%s: Unknown action param '%s'.",
                    $function,
                    $action
                );
                break;
        }

        $this->buildResponse($params);

        return $this->response;
    }

    public function toArray()
    {
        $data = [];

        foreach ($this->getDataFields() as $key => $params) {
            $data[$key] = $this->object->$key;
        }

        return $data;
    }

    public function create($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        if ($this->response->result) {
            if (defined("$this->class::TABLE_KEY")) {
                $params = $this->applyTableConfiguration(
                    $this->class::TABLE_KEY,
                    $params,
                    $this->response->errors
                );
            }

            // ------------------------------------------
            // Check if we were given an Array of Data rows
            // If not, convert to an Array of Data rows
            // ------------------------------------------
            if (! self::isArray($params)) {
                $wasArray = false;
                $paramsArray = [$params];
            } else {
                $wasArray = true;
                $paramsArray = $params;
                $this->response->data = [];
            }

            // ------------------------------------------
            // Create Individual Data rows
            // ------------------------------------------
            foreach ($paramsArray as $index => $params) {
                $this->object = new $this->classPath;
                $this->setDefaultValues();
                $primaryKey = $this->getPrimaryKeyName($function);
                if (! empty($params[$primaryKey]) && $params[$primaryKey] == 'new') {
                    unset($params[$primaryKey]);
                }
                $this->populate($params);

                // -----------------------------
                // Run the Query
                // -----------------------------
                if (! empty($params['withSql'])) {
                    DB::enableQueryLog();
                }

                $rc = $this->object->save();

                if (! empty($params['withSql'])) {
                    $queryLogs = DB::getQueryLog();
                    $this->response->sql[] = $queryLogs[0]['query'];
                    DB::disableQueryLog();
                }

                if ($rc) {
                    if ($wasArray) {
                        $this->response->data[] = $this->toArray();
                    } else {
                        $this->response->data = $this->toArray();
                    }
                }

                // -----------------------------
                // Check for Errors
                // -----------------------------
                if (! $rc) {
                    $this->response->result = false;
                    if ($wasArray) {
                        $message = sprintf(
                            "%s: Unable to Create %s for item number %s.",
                            $function,
                            $this->class,
                            number_format($index + 1)
                        );
                    } else {
                        $message = sprintf(
                            "%s: Unable to Create %s.",
                            $function,
                            $this->class
                        );
                    }
                    $this->response->errors[] = $message;
                    break;
                }
            }
        }

        return $this->response;
    }

    public function update($params)
    {
        $function = __FUNCTION__;
        $this->initResponse($params);

        $primaryKey = $this->getPrimaryKeyName($function);
        $requireds  = [$primaryKey];

        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        if (defined("$this->class::TABLE_KEY")) {
            $params = $this->applyTableConfiguration(
                $this->class::TABLE_KEY,
                $params,
                $this->response->errors
            );

            if (!empty($this->response->errors)) {
                $this->response->result = false;
                return $this->response;
            }
        }

        // ------------------------------------------
        // Check if we were given an Array of Data rows
        // If not, convert to an Array of Data rows
        // ------------------------------------------
        if (! self::isArray($params)) {
            $wasArray = false;
            $paramsArray  = [$params];
        } else {
            $wasArray = true;
            $paramsArray  = $params;
        }

        // ------------------------------------------
        // Update Individual Data rows
        // ------------------------------------------
        foreach ($paramsArray as $index => $params) {
            // Find the current Record
            if (! empty($params['withSql'])) {
                DB::enableQueryLog();
            }

            $object = $this->classPath::find($params[$primaryKey]);

            if (! empty($params['withSql'])) {
                $queryLogs = DB::getQueryLog();
                $this->response->sql[] = $queryLogs[0]['query'];
                DB::disableQueryLog();
            }

            if (! $object) {
                $this->response->result = false;
                if ($wasArray) {
                    $message = sprintf(
                        "%s: Can't Find %s for Id = '%s', item number %s.",
                        $function,
                        $this->class,
                        $params['id'],
                        number_format($index + 1)
                    );
                } else {
                    $message = sprintf(
                        "%s: Can't Find %s for Id = '%s'.",
                        $function,
                        $this->class,
                        $params['id']
                    );
                }
                $this->response->errors[] = $message;
                break;
            }

            if ($this->response->result) {
                $this->object = $object;
                $this->populate($params);

                // Update the Record
                if (! empty($params['withSql'])) {
                    DB::enableQueryLog();
                }

                $rc = $this->object->save();

                if (! empty($params['withSql'])) {
                    $queryLogs = DB::getQueryLog();
                    $this->response->sql[] = $queryLogs[0]['query'];
                    DB::disableQueryLog();
                }

                // Check for Errors
                if (! $rc) {
                    $this->response->result = false;
                    if ($wasArray) {
                        $message = sprintf(
                            "%s: Unable to Update %s for Id = '%s', item number %s.",
                            $function,
                            $this->class,
                            $params['id'],
                            number_format($index + 1)
                        );
                    } else {
                        $message = sprintf(
                            "%s: Unable to Update %s for Id = '%s'.",
                            $function,
                            $this->class,
                            $params['id']
                        );
                    }
                    $this->response->errors[] = $message;
                    break;
                }
            }
        }

        return $this->response;
    }

    public function delete($params)
    {
        $function = __FUNCTION__;

        $this->initResponse($params);

        $primaryKey = $this->getPrimaryKeyName($function);
        $requireds  = [$primaryKey];

        if (! $this->checkRequireds($function, $params, $requireds)) {
            return $this->response;
        }

        // ------------------------------------------
        // Check if we were given an Array of Data rows
        // If not, convert to an Array of Data rows
        // ------------------------------------------
        if (! self::isArray($params)) {
            $wasArray   = false;
            $paramsArray  = [$params];
        } else {
            $wasArray   = true;
            $paramsArray = $params;
        }

        // ------------------------------------------
        // Delete Individual Data rows
        // ------------------------------------------
        if ($this->response->result) {
            foreach ($paramsArray as $index => $params) {
                // -----------------------------
                // Find the Record
                // -----------------------------
                if (! empty($params['withSql'])) {
                    DB::enableQueryLog();
                }

                $this->object = $this->classPath::find($params[$primaryKey]);

                if (! empty($params['withSql'])) {
                    $queryLogs = DB::getQueryLog();
                    $this->response->sql[] = $queryLogs[0]['query'];
                    DB::disableQueryLog();
                }

                // -----------------------------
                // Check for Errors
                // -----------------------------
                if (! $this->object) {
                    $this->response->result = false;
                    if ($wasArray) {
                        $message = sprintf(
                            "%s: Can't Find %s for Id = '%s', item number %s.",
                            $function,
                            $this->class,
                            $params[$primaryKey],
                            number_format($index + 1)
                        );
                    } else {
                        $message = sprintf(
                            "%s: Can't Find %s for Id = '%s'.",
                            $function,
                            $this->class,
                            $params[$primaryKey]
                        );
                    }
                    $this->response->errors[] = $message;
                    break;
                }

                // -----------------------------
                // Delete the Record
                // -----------------------------
                if (! empty($params['withSql'])) {
                    DB::enableQueryLog();
                }

                $rc = $this->object->delete();

                if (! empty($params['withSql'])) {
                    $queryLogs = DB::getQueryLog();
                    $this->response->sql[] = $queryLogs[0]['query'];
                    DB::disableQueryLog();
                }

                // -----------------------------
                // Check for Errors
                // -----------------------------
                if (! $rc) {
                    $this->response->result = false;
                    if ($wasArray) {
                        $message = sprintf(
                            "%s: Unable to Delete %s, item number %s.",
                            $function,
                            $this->class,
                            number_format($index + 1)
                        );
                    } else {
                        $message = sprintf(
                            "%s: Unable to Delete %s.",
                            $function,
                            $this->class
                        );
                    }
                    $this->response->errors[] = $message;
                    break;
                }
            }
        }

        return $this->response;
    }

    public function sync($dataItems, $class, $foreignKeyName, $foreignKeyValue = false)
    {
        $function = __FUNCTION__;

        $this->initResponse($dataItems);

        // Make sure we have an Array of items
        // Note: empty() check below is a special case for deleting all of the dataItems
        if (! empty($dataItems) && ! self::isArray($dataItems)) {
            $message = sprintf(
                "%: Data provided for '%s' is not an array.",
                $function,
                $class
            );
            $this->response->errors[] = $message;
            return $this->response;
        }

        // Make sure we have the required params
        $first = array_first($dataItems);
        $primaryKey = $this->getPrimaryKeyName($function);

        // Deal with Special Case for deleting all dataItems
        if (! empty($dataItems)) {
            $requireds = [$primaryKey, $foreignKeyName];
            if (! $this->checkRequireds($function, $first, $requireds)) {
                return $this->response;
            }
            $foreignKeyValue = ($foreignKeyValue === false) ? $first[$foreignKeyName] : $foreignKeyValue;
        } else {
            if ($foreignKeyValue === false) {
                $message = sprintf(
                    "%: foreignKey required if '%s' Data array is empty.",
                    $function,
                    $class
                );
                $this->response->errors[] = $message;
                return $this->response;
            }
        }

        // Get current Items
        $params = [
            'mode' => 'array',
            'class' => $class,
            'action' => 'getList',
            $foreignKeyName => $foreignKeyValue
        ];
        $response   = $this->get($params);

        if (! $response->result) {
            $message = sprintf(
                "%s: Unable to get Current %s records for %s = '%s'.",
                $function,
                $this->class,
                $foreignKeyName,
                $foreignKeyValue
            );
            $this->response->errors[] = $message;
            return $this->response;
        }

        $currentItems = $response->data;

        // -----------------------------------------
        // Create Item lookup arrays
        // -----------------------------------------
        $itemsLookup = [];
        $currentItemsLookup = [];

        foreach ($dataItems as $item) {
            // --------------
            // Make sure we have the required params
            // --------------
            if (! $this->checkRequireds($function, $item, $requireds)) {
                return $this->response;
            }

            if (isset($item[$primaryKey]) && $item[$primaryKey] != 'new') {
                $itemsLookup[$item[$primaryKey]] = $item;
            }
        }

        foreach ($currentItems as $item) {
            $currentItemsLookup[$item[$primaryKey]] = $item;
        }

        // -----------------------------------------
        // Get lists of Items to Add or Update
        // -----------------------------------------
        $addItems       = [];
        $updateItems    = [];
        $deleteItems    = [];

        // -------------------
        // Find Adds / Updates
        // -------------------
        foreach ($dataItems as $item) {
            $primaryKeyValue = $item[$primaryKey];

            $item['class'] = $class;

            if (! isset($item[$primaryKey]) || $item[$primaryKey] == 'new') {
                // Check if this is a new item
                $item[$foreignKeyName] = $foreignKeyValue;
                $item['action'] = 'create';
                $addItems[] = $item;
            } elseif (isset($currentItemsLookup[$primaryKeyValue])) {
                // Check if this is an existing item
                $item['action'] = 'update';
                $updateItems[] = $item;
            }
        }

        // -------------------
        // Find Items to Delete
        // -------------------
        foreach ($currentItems as $item) {
            $primaryKeyValue = $item[$primaryKey];

            $item['class']  = $class;
            $item['action'] = 'delete';

            if (! isset($itemsLookup[$primaryKeyValue])) {
                $deleteItems[] = $item;
            }
        }

        // Handle Special case where all Items have been Deleted
        if (empty($itemsLookup)) {
            $deleteItems = $currentItems;
        }

        // -----------------------------------------
        // Delete Removed Items
        // Need to do this first, in case of possible Unique Index collisions
        // -----------------------------------------
        if (count($deleteItems) > 0) {
            $response = $this->delete($deleteItems);
 
            if (! $response->result) {
                $message = sprintf(
                    "%s: Unable to Delete removed %s records for %s = '%s'.",
                    $function,
                    $this->class,
                    $foreignKeyName,
                    $foreignKeyValue
                );
                array_unshift($this->response->errors, $message);
                return $response;
            }
        }

        // -----------------------------------------
        // Update Existing Items
        // -----------------------------------------
        if (count($updateItems) > 0) {
            $response = $this->update($updateItems);
 
            if (! $response->result) {
                $message = sprintf(
                    "%s: Unable to Update existing %s records for %s = '%s'.",
                    $function,
                    $this->class,
                    $foreignKeyName,
                    $foreignKeyValue
                );
                array_unshift($this->response->errors, $message);
                return $response;
            }
        }

        // -----------------------------------------
        // Add New Items
        // -----------------------------------------
        if (count($addItems) > 0) {
            $response = $this->create($addItems);
 
            if (! $response->result) {
                $message = sprintf(
                    "%s: Unable to add new %s records for %s = '%s'.",
                    $function,
                    $this->class,
                    $foreignKeyName,
                    $foreignKeyValue
                );
                array_unshift($this->response->errors, $message);
                return $response;
            }
        }

        return $this->response;
    }
}
