<?php

namespace Ignite\Claim\Providers;

use Ignite\Core\Facades\Format;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Database\Eloquent\Factory;
use Illuminate\Support\ServiceProvider;
use Ignite\Claim\Models\CapCheck;
use Ignite\Claim\Repositories\ClaimRepository;
use Ignite\Claim\Repositories\OfferRepository;
use Ignite\Claim\Repositories\SettingsRepository;
use Ignite\Core\Repositories\TransactionRepository;

class ClaimServiceProvider extends ServiceProvider
{
    /**
     * The module name.
     *
     * @var string
     */
    protected $moduleName = 'Claim';

    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var bool
     */
    protected $defer = false;

    /**
     * Indicates whether the views for the module can be published to the vendor directory.
     *
     * @var bool
     */
    protected $allowPublishViews = false;

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->bootConfig();
        $this->bootViews();
        $this->bootTranslations();
        $this->bootFactories();
        $this->bootAssets();
        $this->loadMigrationsFrom(__DIR__ . '/../Database/Migrations');
    }

    /**
     * Publish assets for ignite.
     */
    protected function bootAssets()
    {
        $this->publishes([
            __DIR__.'/../Assets/ignite' => public_path('vendor/ignite'),
        ], 'assets');
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function bootConfig()
    {
        $configPath = __DIR__ . '/../Config/config.php';
        $configName = strtolower($this->moduleName);

        $this->publishes(["$configPath" => config_path($configName . '.php')], 'config');

        if (file_exists($configPath)) {
            $this->mergeConfigFrom($configPath, $configName);
        }
    }

    /**
     * Register views.
     *
     * @return void
     */
    public function bootViews()
    {
        $sourcePath = __DIR__ . '/../Resources/views';

        if ($this->allowPublishViews) {
            $this->publishes([$sourcePath => base_path('resources/views/modules/' . $this->moduleName)]);
        }

        $this->loadViewsFrom(array_merge(array_map(function ($path) {
            return $path . '/modules/' . $this->moduleName;
        }, $this->app['config']->get('view.paths')), [$sourcePath]), $this->moduleName);
    }

    /**
     * Register translations.
     *
     * @return void
     */
    public function bootTranslations()
    {
        $langPath = base_path('resources/lang/modules/' . strtolower($this->moduleName));

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleName);
        } else {
            $this->loadTranslationsFrom(__DIR__ . '/../Resources/lang', $this->moduleName);
        }
    }

    /**
     * Register an additional directory of factories.
     *
     * @return void
     */
    public function bootFactories()
    {
        if (! app()->environment('production')) {
            app(Factory::class)->load(__DIR__ . '/../Database/Factories');
        }
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        $this->app->register(RouteServiceProvider::class);
        $this->app->register(EventServiceProvider::class);
        $this->app->register(ViewServiceProvider::class);

        $this->app->singleton(CapCheck::class, function (Application $app) {
            return new CapCheck(
                $app->make(ClaimRepository::class),
                $app->make(OfferRepository::class),
                $app->make(TransactionRepository::class)
            );
        });

        $this->app->bind('settings', function (Application $app) {
            return $app->make(SettingsRepository::class);
        });
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
