<?php

namespace Ignite\Claim\Models\Legacy;

use Ignite\Claim\Entities\Table;

class TableConfigService
{
    protected $errors = [];

    /**
     * Apply the table configuration to the submitted form data.
     *
     * @param  array $keys
     * @param  array $formData
     * @return array
     */
    public function apply(array $keys, $formData)
    {
        foreach ($keys as $key) {
            /** @var TableConfig $config */
            foreach ($this->getTableConfig($key) as $field => $config) {
                if (isset($formData[$field])) {
                    $value = is_array($formData[$field]) ? implode(',', $formData[$field]) : $formData[$field];
                    try {
                        $formData = $this->applyTypeNegotiation($formData, $field, $config);
                        $formData = $this->applyEncryption($formData, $field, $config);
                    } catch (\Exception $e) {
                        $this->errors[$field] = sprintf(
                            "The field %s could not be saved because value '%s' generated an error: %s",
                            $field,
                            $value,
                            $e->getMessage()
                        );
                    }
                }
            }
        }

        return $formData;
    }

    /**
     * Determine if the service generated errors.
     *
     * @return bool
     */
    public function hasErrors()
    {
        return ! empty($this->errors);
    }

    /**
     * Get the errors generated by the service.
     *
     * @return array
     */
    public function getErrors()
    {
        return $this->errors;
    }

    /**
     * Apply the type negotiation.
     *
     * @param  array $formData
     * @param  string $field
     * @param  TableConfig $config
     * @return array|mixed
     */
    protected function applyTypeNegotiation(array $formData, $field, TableConfig $config)
    {
        /** @var TypeNegotiator $typeNegotiator */
        $typeNegotiator = app(TypeNegotiator::class, [
            'value' => $formData[$field],
            'config' => $config
        ]);

        $formData[$field] = $typeNegotiator->apply($field);

        return $formData;
    }

    /**
     * Apply encryption on sensitive fields.
     *
     * @param  array $formData
     * @param  string $field
     * @param  TableConfig $config
     * @return array
     */
    protected function applyEncryption(array $formData, $field, TableConfig $config)
    {
        if ($config->isSensitive() && $config->get('type') !== 'password') {
            $formData[$field] = encrypt($formData[$field]);
        }

        return $formData;
    }

    /**
     * Get the table configuration model as a collection of field => TableConfig models.
     *
     * @param  string $key
     * @return \Illuminate\Support\Collection
     */
    protected function getTableConfig($key)
    {
        $tableSettings = Table::findByKey($key);

        if (empty($tableSettings)) {
            return collect();
        }

        $columns = $tableSettings->getMergedColumns();

        if (empty($columns)) {
            return collect();
        }

        return collect($columns)->mapInto(TableConfig::class);
    }
}
