<?php

namespace Ignite\Claim\Http\Controllers\Api;

use App;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use Ignite\Claim\Entities\Exception;
use Ignite\Claim\Entities\Offer;
use Ignite\Claim\Entities\OfferCampaign;
use Ignite\Claim\Entities\OfferPromotion;
use Ignite\Claim\Entities\OfferPromotionGroup;
//use Ignite\Claim\Entities\OfferPromotionUserGroup;
use Ignite\Claim\Entities\OfferUserPromotionGroup;
use Ignite\Claim\Entities\OfferUser;
use Ignite\Claim\Repositories\OfferRepository;

class OfferController extends Controller
{
    private $offerRepository;

    /**
     * Create a new controller instance.
     *
     * @param OfferRepository $offerRepository
     */
    public function __construct(OfferRepository $offerRepository)
    {
        $this->offerRepository = $offerRepository;
    }

    // ------------------------------------------------------
    // Campaign Methods
    // ------------------------------------------------------

    public function campaignGet($campaignId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $response->data = OfferCampaign::find($campaignId);
        } catch (Exception $e) {
            $response->result       = false;
            $response->data         = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function campaignStore()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            if ($data['id'] == 'new') {
                $campaign = new OfferCampaign;
            } else {
                $campaign = OfferCampaign::find($data['id']);
            }

            $startDate  = (! empty($data['start_date']))  ? date('Y-m-d', strtotime($data['start_date'])) : null;
            $endDate    = (! empty($data['end_date']))    ? date('Y-m-d', strtotime($data['end_date'])) : null;

            $campaign->name        = $data['name'];
            $campaign->start_date  = $startDate;
            $campaign->end_date    = $endDate;
            $campaign->description = $data['description'];
            $campaign->status      = $data['status'];

            $rc = $campaign->save();

            if (! $rc) {
                $response->result   = false;

                if ($data['id'] == 'new') {
                    $message = sprintf("Unable to Create new Campaign.\n");
                } else {
                    $message = sprintf("Unable to Update Campaign Id = '%s'.\n", $data['id']);
                }

                throw new Exception($message);
            }
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function campaignDelete()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $campaign = OfferCampaign::find($data['id']);

            $campaign->delete();
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function getCampaignPromotions($campaignId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        $response->data = new \stdClass;

        $response->data->campaign              = null;
        $response->data->notHasPromotions  = [];
        $response->data->hasPromotions     = [];

        // --------------------------------
        // Add Campaign Info to Response
        // --------------------------------
        $campaign = OfferCampaign::find($campaignId);

        // Save campaign info before lazy load of promotions, or all will be sent on ajax response
        $response->data->campaign     = $campaign->toArray();

        // --------------------------------
        // Get Promotions that already on this Campaign
        // --------------------------------
        $havePromotions = $campaign->promotions;

        foreach ($havePromotions as $promotion) {
            $tmpPromotion = [];
            $tmpPromotion['id']     = $promotion->id;
            $tmpPromotion['name']   = $promotion->name;

            $response->data->hasPromotions[$promotion->id]  = $tmpPromotion;
        }

        // --------------------------------
        // Get Promotions that are not on this Campaign
        // --------------------------------
        $allPromotions = OfferPromotion::all();

        foreach ($allPromotions as $promotion) {
            if (! isset($response->data->hasPromotions[$promotion->id])) {
                $tmpPromotion = [];
                $tmpPromotion['id']     = $promotion->id;
                $tmpPromotion['name']   = $promotion->name;

                $response->data->notHasPromotions[$promotion->id]  = $tmpPromotion;
            }
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    // ------------------------------------------------------
    // Promotion Methods
    // ------------------------------------------------------

    public function promotionGet($promotionId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $response->data = OfferPromotion::with('caps')->find($promotionId);
        } catch (Exception $e) {
            $response->result       = false;
            $response->data         = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionGetForCampaign($campaignId = 1)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $response->data = OfferPromotion::where('campaign_id', '=', $campaignId)
                ->orderBy('name')
                ->get()
                ->toArray();
        } catch (Exception $e) {
            $response->result       = false;
            $response->data         = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionStore()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            // ---------------------------------
            // Put Dates in DB friendly format
            // ---------------------------------
            $data['start_date']  = (! empty($data['start_date']))  ? date('Y-m-d', strtotime($data['start_date'])) : null;
            $data['end_date']    = (! empty($data['end_date']))    ? date('Y-m-d', strtotime($data['end_date'])) : null;

            // ---------------------------------
            // Create / Update Promotion
            // ---------------------------------
            $offerSvc = $this->offerRepository;
            $data['class']  = 'claim_offer_promotion';

            if ($data['id'] == 'new') {
                $data['action']  = 'createPromotion';
            } else {
                $data['action']  = 'updatePromotion';
            }

            $result = $offerSvc->put($data);

            // ---------------------------------
            // Check Response
            // ---------------------------------
            $response->result   = $result->result;
            $response->data     = $result->data;
            $response->messages = $result->errors;

            if (! $response->result) {
                if ($data['id'] == 'new') {
                    $message = sprintf("Unable to Create new Promotion.\n");
                } else {
                    $message = sprintf("Unable to Update Promotion Id = '%s'.\n", $data['id']);
                }

                throw new Exception($message);
            }
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionDelete()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $promotion = OfferPromotion::find($data['id']);

            $promotion->delete();
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    // ------------------------------------------------------
    // Promotion Group Methods
    // ------------------------------------------------------

    public function promotionGroupGet($promotionGroupId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $response->data = OfferPromotionGroup::find($promotionGroupId);
        } catch (Exception $e) {
            $response->result       = false;
            $response->data         = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionGroupStore()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            if ($data['id'] == 'new') {
                $promotionGroup = new OfferPromotionGroup;
            } else {
                $promotionGroup = OfferPromotionGroup::find($data['id']);
            }

            $promotionGroup->name           = $data['name'];
            $promotionGroup->description    = $data['description'];
            $promotionGroup->status         = $data['status'];

            $rc = $promotionGroup->save();

            if (! $rc) {
                $response->result   = false;

                if ($data['id'] == 'new') {
                    $message = sprintf("Unable to Create new Promotion Group.\n");
                } else {
                    $message = sprintf("Unable to Update Promotion Group Id = '%s'.\n", $data['id']);
                }

                throw new Exception($message);
            }
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionGroupDelete()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $promotionGroup = OfferPromotionGroup::find($data['id']);

            $promotionGroup->delete();
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    // ------------------------------------------------------
    // Offer Methods
    // ------------------------------------------------------

    public function offerGet($offerId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            $offerSvc = $this->offerRepository;

            $params = [
                'class' => 'claim_offer',
                'action' => 'find',
                'mode' => 'array',
                'with' => ['values'],
                'id' => $offerId
            ];
            $result = $offerSvc->get($params);

            $response->result       = $result->result;
            $response->data         = $result->data;
            $response->messages     = $offerSvc->svc_errors;
        } catch (Exception $e) {
            $response->result       = false;
            $response->data         = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function offerStore()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        // Validate Request Data
        $messages = $this->validateOffer($data);

        if (! empty($messages)) {
            $response->result   = false;
            $response->messages = $messages;
            $httpCode = 422;
            return response(json_encode($response), $httpCode);
        }

        try {
            // Put Dates in DB friendly format
            $data['start_date']  = (! empty($data['start_date']))  ? date('Y-m-d', strtotime($data['start_date'])) : null;
            $data['end_date']    = (! empty($data['end_date']))    ? date('Y-m-d', strtotime($data['end_date'])) : null;
            $data['grace_date']  = (! empty($data['grace_date']))  ? date('Y-m-d', strtotime($data['grace_date'])) : null;

            $data['value'] = (float) str_replace(',', '', $data['value']);
            $data['value'] = ($data['value_mode_id'] == 1) ? $data['value'] : 0;


            // Update Offer
            $data['class']  = 'claim_offer';

            if ($data['id'] == 'new') {
                $data['action']  = 'createOffer';
            } else {
                $data['action']  = 'updateOffer';
            }

            $result = $this->offerRepository->put($data);

            // Check Response
            $response->result   = $result->result;
            $response->data     = $result->data;
            $response->messages = $result->errors;

            if (! $response->result) {
                if ($data['id'] == 'new') {
                    $message = sprintf("Unable to Create new Offer.\n");
                } else {
                    $message = sprintf("Unable to Update Offer Id = '%s'.\n", $data['id']);
                }

                throw new Exception($message);
            }
        } catch (\Exception $e) {
            $response->result   = false;
            $response->messages[] = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function offerDelete()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        try {
            // ---------------------------------
            // Delete Offer
            // ---------------------------------
            $offerSvc = $this->offerRepository;
            $data['class']  = 'claim_offer';
            $data['action'] = 'deleteOffer';

            $result = $offerSvc->put($data);

            // ---------------------------------
            // Check Response
            // ---------------------------------
            $response->result   = $result->result;
            $response->data     = $result->data;
            $response->messages = $result->errors;

            if (! $response->result) {
                $message = sprintf("Unable to Delete Offer for Id = '%s'.\n", $data['id']);

                throw new Exception($message);
            }
        } catch (\Exception $e) {
            $response->result       = false;
            $response->messages[]   = $e->getMessage();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionGroupGetForUser($userId)
    {
        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        $response->data = new \stdClass;

        $response->data->user           = null;
        $response->data->notHasGroups   = [];
        $response->data->hasGroups      = [];

        // --------------------------------
        // Add User Info to Response
        // --------------------------------
        $user = OfferUser::find($userId);

        // Save user info before lazy load of groups, or all will be sent on ajax response
        $response->data->user     = $user->toArray();

        // --------------------------------
        // Get Groups that User already has
        // --------------------------------
        $haveGroups = $user->promotionGroups;

        foreach ($haveGroups as $group) {
            $tmpGroup = [];
            $tmpGroup['id']    = $group->id;
            $tmpGroup['name']  = $group->name;

            $response->data->hasGroups[$group->id]  = $tmpGroup;
        }

        // --------------------------------
        // Get Groups that User doesn't have
        // --------------------------------
        $allGroups = OfferPromotionGroup::all();

        foreach ($allGroups as $group) {
            if (! isset($response->data->hasGroups[$group->id])) {
                $tmpGroup = [];
                $tmpGroup['id']    = $group->id;
                $tmpGroup['name']  = $group->name;

                $response->data->notHasGroups[$group->id]  = $tmpGroup;
            }
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    public function promotionGroupStoreForUser()
    {
        $data = request()->all();

        $response = new \stdClass;
        $response->result   = true;
        $response->messages = [];

        $response->data = new \stdClass;

        $userId          = $data['userId'];
        $data['hasGroups']  = (! empty($data['hasGroups'])) ? $data['hasGroups'] : []; // Handle empty set

        $user = OfferUser::with('promotionGroups')->find($userId);

        // --------------------------------
        // Initialize results data
        // --------------------------------
        $hasGroups      = [];
        $addGroups      = [];
        $deleteGroups   = [];

        // --------------------------------
        // Get Groups that already have this Permission
        // --------------------------------
        foreach ($user->promotionGroups as $promotionGroup) {
            $hasGroups[$promotionGroup->id]  = $promotionGroup->name;
        }

        $response->data->hadGroups = array_keys($hasGroups);

        // --------------------------------
        // Determine what Groups we need to add
        // --------------------------------
        $newGroupsLookup = [];
        foreach ($data['hasGroups'] as $groupId => $name) {
            $newGroupsLookup[$groupId] = $name;
            if (! isset($hasGroups[$groupId])) {
                $addGroups[$groupId] = $name;
            }
        }

        $response->data->addedGroups = $addGroups;

        if (count($addGroups) > 0) {
            foreach ($addGroups as $groupId => $name) {
                $userPromotionGroup = new OfferUserPromotionGroup;

                $userPromotionGroup->user_id            = $userId;
                $userPromotionGroup->promotion_group_id     = $groupId;
                $userPromotionGroup->updated_at             = null;

                $userPromotionGroup->save();
            }
        }

        // --------------------------------
        // Determine what Groups we need to delete
        // --------------------------------
        foreach ($hasGroups as $groupId => $name) {
            if (! isset($newGroupsLookup[$groupId])) {
                $deleteGroups[$groupId] = $name;
            }
        }

        $response->data->deletedGroups = $deleteGroups;

        if (count($deleteGroups) > 0) {
            $userGroup = new OfferUserPromotionGroup;
            $userGroup
                ->where('user_id', '=', $userId)
                ->whereIn('promotion_group_id', array_keys($deleteGroups))
                ->delete();
        }

        $httpCode = ($response->result) ? 200 : 422;
        return response(json_encode($response), $httpCode);
    }

    private function validateOffer($data)
    {
        $messages = [];

        // ---------------------------------
        // Do Laravel Validations
        // ---------------------------------
        $rules = [
            'promotion_id'      => 'required|integer',
            'name'              => 'required',
            'value_type_id'     => 'required|integer',
            'value_mode_id'     => 'required|integer',
            // 'value'             => 'requiredif|value_mode_id,1|integer',
        ];

        $validator = validator($data, $rules);

        if ($validator->fails()) {
            $messages = $validator->errors()->all();
        }

        // ---------------------------------
        // Do Custom Validations
        // ---------------------------------
        $data['value'] = (! empty($data['value'])) ? $data['value'] : 0;
        $data['value'] = (float) str_replace(',', '', $data['value']);

        if ($data['value_mode_id'] == 1) {
            if ($data['value'] == 0) {
                $messages[] = sprintf("A value is required when Value Mode is set to Single.");
            }
        } else {
            if (count($data['values']) < 1) {
                $messages[] = sprintf("You must have at least 1 Value by Participant if Value Mode is Multiple.");
            }
        }

        return $messages;
    }
}
