<?php

namespace Ignite\Claim\Entities\Dynamic;

use Illuminate\Support\Facades\DB;

class MariaDB
{
    /** @var array */
    private static $databaseTablesArray = [];

    /** @var array */
    private static $tables = [];

    /** @var array */
    private static $tablesArray = [];

    /** @var array */
    private static $typesMap = [
        'char'      => 'string',
        'varchar'   => 'string',
        'int'       => 'int',
        'smallint'  => 'int',
        'tinyint'   => 'int',
        'double'    => 'double',
        'date'      => 'date',
        'datetime'  => 'datetime',
        'timestamp'  => 'timestamp',
        'blob'      => 'blob',
        'text'      => 'text',
    ];

    /** @var array */
    public static $columnMap = [
        'name'           => 'DB:COLUMN_NAME',
        'displayName'    => 'MAP:name:ucwords',
        'reportName'     => 'MAP:name:ucwords',
        'type'           => 'DB:DATA_TYPE:map:typesMap',
        'null'           => 'DB:IS_NULLABLE:bool:YES,1|NO,0',
        'required'       => 'MAP:null:!',
        'format'         => '',
        'validate'       => '',
    ];

    /**
     * Create a unique key based on the database name and table name.
     *
     * @param  string $databaseName
     * @param  string $tableName
     * @return string
     */
    private static function createKey($databaseName, $tableName)
    {
        return $databaseName . '_' . $tableName;
    }

    /**
     * Get a list of tables.
     *
     * @param  string $databaseName
     * @return mixed
     */
    public static function getTables($databaseName)
    {
        if (! isset(self::$databaseTablesArray[$databaseName])) {
            $tmpTables = DB::connection()
                ->table('information_schema.columns')
                ->select('table_name')
                ->distinct()
                ->where('table_schema', '=', $databaseName)
                ->orderBy('table_name')
                ->get();

            foreach ($tmpTables as $table) {
                self::$databaseTablesArray[$databaseName][] = $table->table_name;
            }
        }

        return self::$databaseTablesArray[$databaseName];
    }

    /**
     * Get a list of columns for the table.
     *
     * @param  string $databaseName
     * @param  string $tableName
     * @param  bool $asArray
     * @return mixed
     */
    public static function getColumns($databaseName, $tableName, $asArray = true)
    {
        $key = self::createKey($databaseName, $tableName);

        if (! isset(self::$tables[$key])) {
            $tmpColumns = DB::connection()
                ->table('information_schema.columns')
                ->where('table_schema', '=', $databaseName)
                ->where('table_name', '=', $tableName)
                ->orderBy('COLUMN_NAME')
                ->get();

            foreach ($tmpColumns as $column) {
                $name = $column->COLUMN_NAME;
                self::$tables[$key][$name] = $column;
                self::$tablesArray[$key][$name] = (array) $column;
            }
        }

        if (empty(self::$tables) || empty(self::$tablesArray)) {
            return false;
        }

        return $asArray ? self::$tablesArray[$key] : self::$tables[$key];
    }

    /**
     * Get a column.
     *
     * @param  string $databaseName
     * @param  string $tableName
     * @param  string $columnName
     * @param  bool   $asArray string
     * @return bool|mixed
     */
    public static function getColumn($databaseName, $tableName, $columnName, $asArray = true)
    {
        $columns = self::getColumns($databaseName, $tableName);

        if (! $columns) {
            return false;
        }

        $key = self::createKey($databaseName, $tableName);

        if (! empty($columns[$columnName])) {
            if ($asArray) {
                return self::$tablesArray[$key];
            } else {
                return self::$tables[$key];
            }
        }

        return false;
    }

    /**
     * Convert a column to database.
     *
     * @param  string $column
     * @param  bool $asArray
     * @return array
     */
    public static function convertColumnToDb($column, $asArray = true)
    {
        $dbColumn = [];

        // Required Fields
        $dbColumn['name'] = $column['COLUMN_NAME'];
        $dbColumn['type'] = self::$typesMap[$column['DATA_TYPE']];
        $dbColumn['null'] = ($column['IS_NULLABLE'] == 'YES') ? '1' : '0';
        $dbColumn['displayName'] = ucwords(str_replace('_', ' ', $column['COLUMN_NAME']));
        $dbColumn['reportName'] = $dbColumn['displayName'];
        $dbColumn['required'] = (! $dbColumn['null']) ? '1' : '0';
        $dbColumn['format'] = '';
        $dbColumn['validate'] = (! $dbColumn['null']) ? 'required' : '';

        if (! $asArray) {
            $dbColumn = (object) $dbColumn;
        }

        return $dbColumn;
    }

    /**
     * Get a Database column.
     *
     * @param  string $databaseName
     * @param  string $tableName
     * @param  string $columnName
     * @return array|bool
     */
    public static function getDbColumn($databaseName, $tableName, $columnName)
    {
        $column = self::getColumn($databaseName, $tableName, $columnName);

        if (empty($column)) {
            return false;
        }

        $dbColumn = self::convertColumnToDb($column);

        return $dbColumn;
    }

    /**
     * Get all Database columns.
     *
     * @param string $databaseName
     * @param string $tableName
     * @param bool $asArray
     * @return array|bool
     */
    public function getDbColumns($databaseName, $tableName, $asArray = true)
    {
        $databaseColumns = self::getColumns($databaseName, $tableName, $asArray);

        if (empty($databaseColumns)) {
            return false;
        }

        $dbColumns = [];

        foreach ($databaseColumns as $name => $params) {
            $dbColumns[$name] = self::convertColumnToDb($params);
        }

        return $dbColumns;
    }
}
