<?php

namespace Ignite\Claim\Entities\Dynamic;

use Ignite\Claim\Entities\Setting;
use Ignite\Claim\Entities\Table;
use Ignite\Claim\Exceptions\DynamicColumnLoadException;

class DynamicColumns
{
    const NOT_CHECKED = -1;

    /** @var bool|string */
    private $dynamicColumnName = 'idc';

    /** @var string */
    private $settingKey = '';

    /** @var bool|array */
    private $columnSettings = false;

    /** @var bool|string */
    private $IgniteDb = false;

    /** @var bool|string */
    private $datbaseName = false;

    /** @var bool|string */
    private $tableName = false;

    /** @var bool|string */
    private $dbColumns = false;

    /** @var int */
    private $hasDynamicColumns = self::NOT_CHECKED;

    /** @var bool|string */
    private $dynamicColumn = false;

    /** @var bool|string */
    private $dynamicColumns = false;

    /** @var bool|array */
    private $dynamicColumnData = false;

    /** @var bool|string */
    private $databaseName;

    /**
     * DynamicColumns constructor.
     *
     * @param string $IgniteDb
     * @param string $tableName
     * @param bool   $databaseName
     * @param bool   $dynamicColumnName
     */
    public function __construct($IgniteDb, $tableName, $databaseName = false, $dynamicColumnName = false)
    {
        $this->IgniteDb = $IgniteDb;
        $this->databaseName = ($databaseName) ? $databaseName : config('database.default', 'undefined');
        $this->tableName = $tableName;
        $this->dynamicColumnName = ($dynamicColumnName)  ? $dynamicColumnName  : $this->dynamicColumnName;

        $this->getDynamicColumn();
    }

    /**
     * Get column settings
     *
     * @return array|bool|mixed
     * @throws DynamicColumnLoadException
     */
    public function getColumnSettings()
    {
        if ($this->columnSettings !== false) {
            return $this->columnSettings;
        }

        $this->settingKey = 'table.columns.' . $this->tableName;
        //$setting = setting()->byKey($this->settingKey);
        $table = Table::findByKey($this->settingKey);

        if (! $table) {
            $errors = [];
            $errors[] = sprintf(
                "No settings entry created for Table = '%s', Key = '%s', can't continue.",
                $this->tableName,
                $this->settingKey
            );
            $errors[] = sprintf(
                "Please use the UI Admin->Settings->Tables List, to Create Settings for this Table.\n"
            );
            throw new DynamicColumnLoadException($errors);
        }

        $this->columnSettings = $table;

        return $this->columnSettings;
    }

    /**
     * Get database columns.
     *
     * @return array|bool|string
     */
    public function getDbColumns()
    {
        if ($this->dbColumns !== false) {
            return $this->dbColumns;
        }

        $table = $this->getColumnSettings();

        // Extract the Database (Static) Columns
        $this->dbColumns = [];
        foreach ($table->static_columns as $name => $staticColumn) {
            $this->dbColumns[$name] = (array) $staticColumn;
        }

        return $this->dbColumns;
    }

    /**
     * Get dynamic columns.
     *
     * @return array|bool|string
     */
    public function getDynamicColumns()
    {
        if ($this->dynamicColumns !== false) {
            return $this->dynamicColumns;
        }

        $table = $this->getColumnSettings();

        // Extract the Dynamic Columns
        // The Table->dynamicColumns can be missing if empty due to JSON not saving it
        $tableDynamicColumns = (! empty($table->dynamic_columns))
            ? $table->dynamic_columns
            : new \stdClass;

        $this->dynamicColumns = [];
        foreach ($tableDynamicColumns as $name => $dynamicColumn) {
            $this->dynamicColumns[$name] = (array) $dynamicColumn;
        }

        return $this->dynamicColumns;
    }

    /**
     * Get a dynamic column.
     *
     * @return bool|string
     */
    public function getDynamicColumn()
    {
        if (! $this->dynamicColumn && $this->hasDynamicColumns == self::NOT_CHECKED) {
            $this->dynamicColumn = $this->IgniteDb->getColumn(
                $this->databaseName,
                $this->tableName,
                $this->dynamicColumnName
            );

            if ($this->dynamicColumn) {
                $this->hasDynamicColumns = true;
            } else {
                $this->hasDynamicColumns = false;
            }
        }

        return $this->dynamicColumn;
    }

    /**
     * Table has a dynamic column.
     *
     * @return int
     */
    public function hasDynamic()
    {
        if ($this->hasDynamicColumns == self::NOT_CHECKED) {
            $this->getDynamicColumn();
        }

        return $this->hasDynamicColumns;
    }

    /**
     * Get the dynamic column name.
     *
     * @return bool|string
     */
    public function getDynamicColumnName()
    {
        return $this->dynamicColumnName;
    }

    /**
     * Determine if the given column name is dynamic.
     *
     * @param  string $name
     * @return bool
     */
    public function isDynamicColumn($name)
    {
        if (! $this->hasDynamicColumns) {
            return false;
        }

        $dynamicColumns = $this->getDynamicColumns();

        if (isset($dynamicColumns[$name])) {
            return true;
        }

        return false;
    }

    /**
     * Get column data.
     *
     * @param  object $object
     * @return array|bool|mixed
     */
    private function getColumnData(&$object)
    {
        if (! $this->hasDynamicColumns) {
            return false;
        }

        $dynamicColumnName = $this->dynamicColumnName;
        if (empty($this->dynamicColumnData)) {
            $dynamicColumnData = (! empty($object->$dynamicColumnName)) ? $object->$dynamicColumnName : '{}';
            $this->dynamicColumnData = json_decode($dynamicColumnData);
        }

        return $this->dynamicColumnData;
    }

    /**
     * Get a column by name.
     *
     * @param  object $object
     * @param  string $name
     * @param  mixed  $default
     * @return null
     */
    public function getColumn(&$object, $name, $default = null)
    {
        if (! $this->hasDynamicColumns) {
            return $default;
        }

        $columnData = $this->getColumnData($object);

        // $name = $this->stripDynamicPrefix($name);

        if (isset($columnData->$name)) {
            return $columnData->$name;
        }

        return $default;
    }

    /**
     * Set column data.
     *
     * @param  object $object
     * @param  array $columnData
     * @return bool
     */
    private function setColumnData(&$object, $columnData)
    {
        if (! $this->hasDynamicColumns) {
            return false;
        }

        $this->dynamicColumnData = $columnData;

        $dynamicColumnName = $this->dynamicColumnName;
        $object->$dynamicColumnName = json_encode($columnData);

        return true;
    }

    /**
     * Set column.
     *
     * @param  object $object
     * @param  string $name
     * @param  mixed $value
     * @return bool
     */
    public function setColumn(&$object, $name, $value)
    {
        $dynamicColumns = $this->getDynamicColumns();

        if (! isset($dynamicColumns[$name])) {
            $message = sprintf(
                "Column = '%s', not defined in Dynamic Columns for Table = '%s', Key ='%s', so it can't be set.",
                $name,
                $this->tableName,
                $this->settingKey
            );
            throw new Exception($message);
        }

        // ------------------------------------
        // Check if no Dynamic Columns have ever been saved yet
        // ------------------------------------
        $columnData = $this->getColumnData($object);
        if (is_null($columnData) || empty($columnData) || ! is_object($columnData)) {
            $columnData = new \stdClass();
        }

        $columnData->$name = $value;

        $this->setColumnData($object, $columnData);

        return true;
    }
}
