<?php

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Migrations\Migration;

class CreateCatalogItemTable extends Migration
{
    protected $tableName = 'catalog_item';

    protected $fulltextColumns = [
        'manufacturer',
        'name',
        'description'
    ];

    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        if (! Schema::hasTable($this->tableName)) {
            Schema::create('catalog_item', function (Blueprint $table) {
                $table->increments('id');
                $table->unsignedInteger('catalog_vendor_id');
                $table->unsignedInteger('catalog_id');
                $table->string('code', 255);
                $table->string('sku', 255);
                $table->string('type', 255);
                $table->string('class', 255);
                $table->string('name', 255);
                $table->string('short_description', 255)->nullable();
                $table->text('description')->nullable();
                $table->text('terms')->nullable();
                $table->string('manufacturer')->nullable();
                $table->string('image', 255)->nullable();
                $table->float('msrp', 100, 4)->default(0)->nullable();
                $table->float('price', 100, 4)->default(0)->nullable();
                $table->float('price_markup', 100, 4)->default(0)->nullable();
                $table->float('price_margin', 100, 4)->default(0)->nullable();
                $table->float('point_value', 100, 4)->default(0)->nullable();
                $table->float('points', 100, 4)->default(0)->nullable();
                $table->float('points_min', 100, 4)->default(0)->nullable();
                $table->float('points_max', 100, 4)->default(0)->nullable();
                $table->integer('visibility')->default(0);
                $table->integer('active')->default(0);
                $table->integer('vendor_active')->default(0);
                $table->json('vendor_meta');
                $table->timestamps();
                $table->foreign('catalog_vendor_id')->references('id')->on('catalog_vendor');
                $table->foreign('catalog_id')->references('id')->on('catalog');
            });

            foreach ($this->fulltextColumns as $column) {
                DB::statement($this->addFulltext([$column], $column));
            }

            DB::statement($this->addFulltext($this->fulltextColumns, 'all'));
            DB::statement("ALTER TABLE catalog_item AUTO_INCREMENT = 100");
        }
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        foreach ($this->fulltextColumns as $column) {
            DB::statement($this->dropFulltext($column));
        }

        DB::statement($this->dropFulltext('all'));

        Schema::drop($this->tableName);
    }

    /**
     * Get an SQL statement to add a fulltext index.
     *
     * @param  array  $columns
     * @param  string $index
     * @return string
     */
    public function addFulltext(array $columns, string $index)
    {
        return sprintf(
            'ALTER TABLE catalog_item ADD FULLTEXT %s(%s)',
            $this->index($index),
            implode(',', $columns)
        );
    }

    /**
     * Get an SQL statement to drop a fulltext index.
     *
     * @param  string $name
     * @return string
     */
    public function dropFulltext($name)
    {
        return sprintf(
            'ALTER TABLE catalog_item DROP INDEX %s',
            $this->index($name)
        );
    }

    /**
     * Create the index name.
     *
     * @param  string $name
     * @return string
     */
    public function index($name)
    {
        return $name . '_ft_idx';
    }
}
