<?php

namespace Ignite\Catalog\Repositories;

use Ignite\Catalog\Collections\BuyRequestCollection;
use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Contracts\CartRepository as CartRepositoryContract;
use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Core\Entities\User;

class CartRepository implements CartRepositoryContract
{
    /**
     * The most basic cart query.
     *
     * @param  array $relations
     * @return \Illuminate\Database\Eloquent\Builder?
     */
    public function query($relations = [])
    {
        return CartItem::with(array_merge(['item', 'super'/*, 'catalog', 'vendor'*/], $relations));
    }

    /**
     * The query scoped for a user.
     *
     * @param  User|null $user
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function queryByUser(User|null $user = null)
    {
        $query = $this->query();

        if ($user instanceof User) {
            return $query->byUser($user);
        }

        return $query->byCurrentUser();
    }

    /**
     * Find cart items by a user. If a user is not provided, use the currently
     * logged in user. Optionally, pass whitelisted item ids to only return
     * those records.
     *
     * @param  User|null $user
     * @param  array     $whitelist
     * @return \Illuminate\Support\Collection
     */
    public function findByUser(User|null $user = null, array $whitelist = [])
    {
        $query = $this->queryByUser($user);

        if (! empty($whitelist)) {
            $query->forItems($whitelist);
        }

        return $query->get();
    }

    /**
     * Find all cart items. Not scoped by user.
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Find a cart item.
     *
     * @param  int       $id
     * @param  string    $column
     * @param  User|null $user
     * @return \Illuminate\Database\Eloquent\Model|CartItem
     */
    public function find($id, $column = 'id', User|null $user = null)
    {
        return $this->queryByUser($user)->where($column, $id)->first();
    }

    /**
     * Find the total points in a user's cart. If the user is not specified, use the currently logged in user.
     *
     * @param  User|null $user
     * @return int
     */
    public function total(User|null $user = null)
    {
        return $this->queryByUser($user)->activeOnly()->sum(DB::raw('quantity * points'));
    }

    /**
     * Add an item to the cart.
     *
     * @param  BuyRequest $buyRequest
     * @return CartItem
     */
    public function add(BuyRequest $buyRequest)
    {
        $col = 'item_id';
        $data = $buyRequest->toArray();
        $item = $buyRequest->getItem()->getKey();

        // If the item is already in the cart, just update the quantity.
        $existing = $this->find($item, $col);
        if ($existing) {
            $data['quantity'] = (int) $existing->quantity + $buyRequest->getQuantity();
        }

        return CartItem::updateOrCreate([$col => $item], $data);
    }

    /**
     * Add an item to the cart as just a single quantity
     *
     * @param  BuyRequest $buyRequest
     * @return CartItem
     */
    public function addSingleQuantity(BuyRequest $buyRequest)
    {
        $data = $buyRequest->toArray();
        $data['quantity'] = 1;

        return CartItem::create($data);
    }

    /**
     * Remove an item from the cart.
     *
     * @param  int $id
     * @return \Illuminate\Database\Eloquent\Model|CartItem
     * @throws \Exception
     */
    public function remove($id)
    {
        $item = $this->find($id, 'item_id');

        if ($item) {
            $item->delete();
        }

        return $item;
    }

    /**
     * Update item quantities by passing along the array of items and quantity pairs:
     * e.g. $updates = [
     *      {item_id} => {quantity}
     * ];
     *
     * @param  BuyRequestCollection $buyRequests
     * @return \Illuminate\Support\Collection
     */
    public function updateQuantities(BuyRequestCollection $buyRequests)
    {
        $items = $this->findByUser(null, $buyRequests->getItemIds()->toArray());

        return $items->filter(function (CartItem $item) use ($buyRequests) {
            return $buyRequests->hasItemId($item->item_id);
        })->map(function (CartItem $item) use ($buyRequests) {
            $buyRequest = $buyRequests->getItemId($item->item_id);
            $quantity = (int) $buyRequest->getQuantity();

            if ($quantity < 1) {
                return $item->delete();
            }

            return $item->update(['quantity' => $quantity]);
        });
    }
}
