<?php

namespace Ignite\Catalog\Models\Grid;

use Ignite\Catalog\Entities\Order;
use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Models\Grid\EloquentTable;
use Illuminate\Database\DatabaseManager;
use Illuminate\Database\Eloquent\Builder;
use Yajra\DataTables\DataTables;

/**
 * @property string status
 */
class OrdersTable extends EloquentTable
{
    /**
     * @var OrderFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    /**
     * OrdersTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param array $params
     * @param OrderFormatter|null $formatter
     */
    public function __construct(
        DataTables      $datatables,
        DatabaseManager $databaseManager,
        OrderFormatter  $formatter,
        array           $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title' => '',
                'data' => 'checkbox',
                'name' => 'checkbox',
                'orderable' => false,
                'searchable' => false,
                'exportable' => false,
                'printable' => false,
                'width' => '10px',
            ],
            'action' => [
                'name' => 'actions',
                'title' => trans('Core::core.common.actions'),
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions',
            ],
            'created_at' => [
                'title' => trans('Catalog::order.fields.created_at.label'),
                'name' => 'catalog_order.created_at',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
            'id' => [
                'title' => trans('Catalog::order.fields.id.label'),
                'name' => 'catalog_order.id',
                'orderable' => true,
                'exportable' => true,
            ],
            'number' => [
                'title' => trans('Catalog::order.fields.number.label'),
                'name' => 'catalog_order.number',
                'orderable' => true,
                'exportable' => true,
            ],
            'user_id' => [
                'title' => trans('Catalog::order.fields.participant.label'),
                'name' => 'catalog_order.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            /*'ship_name' => [
                'title' => trans('Catalog::order.fields.ship_name.label'),
                'name' => 'catalog_order.ship_name',
                'orderable' => true,
                'exportable' => true,
            ],*/
            'ship_email' => [
                'title' => trans('Catalog::order.fields.ship_email.label'),
                'name' => 'catalog_order.ship_email',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_address_1' => [
                'title' => trans('Catalog::order.fields.ship_address_1.label'),
                'name' => 'catalog_order.ship_address_1',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_city' => [
                'title' => trans('Catalog::order.fields.ship_city.label'),
                'name' => 'catalog_order.ship_city',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_state' => [
                'title' => trans('Catalog::order.fields.ship_state.label'),
                'name' => 'catalog_order.ship_state',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_postal' => [
                'title' => trans('Catalog::order.fields.ship_postal.label'),
                'name' => 'catalog_order.ship_postal',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_country' => [
                'title' => trans('Catalog::order.fields.ship_country.label'),
                'name' => 'catalog_order.ship_country',
                'orderable' => true,
                'exportable' => true,
            ],
            'status' => [
                'title' => trans('Catalog::order.fields.status.label'),
                'name' => 'status',
                'orderable' => true,
                'exportable' => true,
                'searchable' => false,
            ],
            'points' => [
                'title' => trans('Catalog::order.fields.points.label'),
                'name' => 'catalog_order.points',
                'orderable' => true,
                'exportable' => true,
            ],
            'quantity' => [
                'title' => trans('Catalog::order.fields.quantity.label'),
                'name' => 'catalog_order.quantity',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The orders query.
     *
     * @return Builder
     */
    public function query(): Builder
    {
        $query = QueryPermissionFilters::for('catalog.order.browse')->apply(
            Order::with('user', 'participant')
        );

        if ($this->status !== 'all') {
            call_user_func([$query, $this->status]);
        }

        return $query;
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'checkbox' => function ($model) { return $this->formatter->checkbox($model); },
            'action' => function ($model) { return $this->formatter->actions($model); },
            'user_id' => function ($model) { return $this->formatter->participant($model); },
            'number' => function ($model) { return $this->formatter->number($model); },
            'ship_email' => function ($model) { return $this->formatter->email($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
            'points' => function ($model) { return $this->formatter->points($model); },
            'created_at' => function ($model) { return $this->formatter->createdAt($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        return parent::getBuilderParameters([
            'order' => [2, 'desc'], // Created At
        ], $this->getCheckboxColumnParameters());
    }
}
