<?php

namespace Ignite\Catalog\Http\Controllers;

use Ignite\Flash\Facades\Flash;
use Illuminate\Http\Request;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Repositories\CatalogRepository;
use Ignite\Core\Contracts\Repositories\PageRepository;
use Ignite\Core\Http\Controllers\Controller;

class CatalogController extends Controller
{
    /**
     * @var CatalogRepository
     */
    private $catalogRepository;

    /**
     * CatalogController constructor.
     *
     * @param CatalogRepository  $catalogRepository
     */
    public function __construct(CatalogRepository $catalogRepository)
    {
        $this->catalogRepository = $catalogRepository;
    }

    /**
     * The catalog index.
     *
     * @param  PageRepository $pageRepository
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function index(PageRepository $pageRepository)
    {
        $catalogs = $this->catalogRepository->findAllActiveForParticipant(auth()->user()->participant);

        $page = $pageRepository->findByCode('catalog');

        if (config('catalog.redirect_when_only_one', 0) && $catalogs->count() === 1) {
            return redirect($catalogs->first()->url());
        }

        return view('catalog.index', compact('catalogs', 'page'));
    }

    /**
     * Show the given catalog.
     *
     * @param  Catalog $catalog
     * @param  Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function show(Catalog $catalog, Request $request)
    {
        if (! session()->has('catalog')) {
            session()->put('catalog', $catalog->toSessionData());
            return redirect()->route('catalog.show', $catalog->code);
        }

        if (! $catalog->allowParticipant(auth()->user())) {
            Flash::error(__('Catalog::theme.catalog.error.unsupported_country'));
            return redirect()->route('catalog.index');
        }

        if (session('catalog.code') !== $catalog->code) {
            session()->put('catalog', $catalog->toSessionData());
            return redirect()->route('catalog.show', $catalog->code);
        }

        $keyword = trim($request->get('keyword', ''));
        $sort = trim($request->get('sort', 'name_asc'));
        $category = trim($request->get('category', ''));
        $perPage = trim($request->get('perpage', config('catalog.per_page')));

        if ($category) {
            session()->put('category', optional(Category::where('id', $category)->first())->toSessionData());
        } else {
            session()->forget('category');
        }

        $items = $this->catalogRepository->search($catalog->getKey(), $keyword, $category, $sort, $perPage);

        return view('catalog.show', compact('catalog', 'items'));
    }
}
