<?php

namespace Ignite\Catalog\Console;

use Ignite\Catalog\Entities\Attribute;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Category;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Entities\Menu;
use Ignite\Catalog\Entities\MenuItem;
use Ignite\Catalog\Entities\Option;
use Ignite\Catalog\Entities\Vendor;
use Ignite\Core\Facades\Format;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

abstract class CatalogItemImportCommandAbstract extends Command
{
    /**
     * Flag to determine if we are in dry run mode.
     *
     * @var bool
     */
    protected $isDryRun;

    /**
     * Positive values for the dry run option.
     *
     * @var array
     */
    protected $dryRunPositiveValues = [
        true, 'true', 1, '1', 'yes', 'y', null
    ];

    /**
     * The collection of categories.
     *
     * @var \Illuminate\Support\Collection
     */
    protected $categories;

    /**
     * Catalogs, keyed by their codes.
     *
     * @var Catalog[]
     */
    protected $catalogs = [];

    /**
     * The catalog menus
     *
     * @var array
     */
    protected $menuIds;

    /**
     * The catalog vendor to attach the items.
     *
     * @var Vendor
     */
    protected $vendor;

    /**
     * @param  string $filename
     * @return array
     */
    abstract protected function getData(string $filename): array;

    /**
     * The catalog vendor entity.
     *
     * @return Vendor
     * @throws \Exception
     */
    abstract protected function vendor();

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $filename = $this->option('file');
        $isReset = $this->option('reset');

        $data = $this->getData($filename);

        if (empty($data)) {
            throw new \DomainException("No data available in $filename.");
        }

        // if $isReset and not on production
        if ($isReset && app()->environment() === 'production') {
            $this->error('Resetting the catalog database on production is not allowed with this.');
            return 1;
        }
        if ($isReset) {
            $this->clear();
        }

        $this->introMessage($data);
        $this->process($data);

        return 0;
    }

    /**
     * @param string     $catalog
     * @param Collection $categories
     */
    protected function addCategoryMenuItems(string $catalog, Collection $categories)
    {
        $catalogMenuId = $this->menuId($catalog);
        foreach ($categories as $category) {
            MenuItem::updateOrCreate([
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
            ], [
                'catalog_menu_id' => $catalogMenuId,
                'catalog_category_id' => $category->id,
                "active" => 1,
                "parent_id" => 0,
                "position" => $category->id,
            ]);
        }
    }

    /**
     * Calculate the cost from the price and markup.
     *
     * @param  float $price
     * @param  float $markup
     * @return float
     */
    protected function calculateCost($price, $markup)
    {
        return $price / (1 + $markup);
    }

    /**
     * Calculate the points from the price and point value.
     *
     * @param  float $usdPrice
     * @param  float $usDollarPerPoint
     * @return int
     */
    protected function calculatePoints($usdPrice, $usDollarPerPoint): int
    {
        return ceil($usdPrice / $usDollarPerPoint);
    }

    /**
     * Calculate the price from the cost and markup.
     *
     * @param  float $cost
     * @param  float $markup
     * @return float
     */
    protected function calculatePrice($cost, $markup)
    {
        return $cost * (1 + $markup);
    }

    /**
     * The catalog entity.
     *
     * @return Catalog
     * @throws \Exception
     */
    protected function catalog($code)
    {
        if (! isset($this->catalogs[$code])) {
            $this->catalogs[$code] = Catalog::where('code', $code)->first();
        }

        if ($this->catalogs[$code]) {
            return $this->catalogs[$code];
        }

        throw new \Exception('No catalogs exist. Run `module:seed` first.');
    }

    /**
     * Get or merge the categories.
     *
     * @param  null|\Illuminate\Support\Collection $items
     * @return static
     */
    protected function categories($items = null)
    {
        if (is_null($this->categories)) {
            $this->categories = collect();
        }

        if (! is_null($items)) {
            foreach ($items as $item) {
                $this->categories->push($item);
            }
        }

        return $this->categories->keyBy('code');
    }

    /**
     * Check the given denomination is close to what is expected based on the cost.
     *
     * @param  string $itemName
     * @param  int    $denomination
     * @param  float  $costUsd
     * @param  float  $exchangeRate
     */
    protected function checkExpectedDenomination(string $itemName, $denomination, $costUsd, float $exchangeRate = 1)
    {
        if (empty($costUsd) || empty($denomination)) {
            return;
        }

        $denominationCalculated = floor($this->exchangeFromUsd($costUsd, $exchangeRate));
        // we will allow a difference of 1 due to rounding
        if (abs($denominationCalculated - $denomination) > 1) {
            $this->warn("Denomination of {$denomination} for {$itemName} is not expected from its cost of \${$costUsd} USD.");
        }
    }

    /**
     * Check the given points is close to what the expected points should be based on the price.
     *
     * @param  string $itemName
     * @param  int    $points
     * @param  float  $price
     */
    protected function checkExpectedPoints(string $itemName, $points, float $price)
    {
        if (empty($points)) {
            return;
        }

        $calculatedPoints = $this->calculatePoints($price, config('catalog.default_point_value'));

        if (abs($points - $calculatedPoints) >= 1) {
            $this->warn(
                "The given points {$points} and calculated points {$calculatedPoints}"
                . " for '{$itemName}' are different."
            );
        }
    }

    /**
     * Check the given price is close to what the expected price should be based on the cost.
     *
     * @param  string $itemName
     * @param  float  $price
     * @param  float  $cost
     * @param  float  $markup
     */
    protected function checkExpectedPrice(string $itemName, float $price, float $cost, float $markup)
    {
        $calculatedPrice = $this->calculatePrice($cost, $markup);
        // we will allow a difference of 1 due to rounding
        if (abs($price - $calculatedPrice) > 1) {
            $this->warn(
                "The given price {$price} and calculated price {$calculatedPrice}"
                . " for '{$itemName}' are different."
            );
        }
    }

    /**
     */
    protected function clear()
    {
        $this->alert('Clearing the catalog item tables!');

        \DB::statement('SET FOREIGN_KEY_CHECKS = 0');
        \DB::delete('TRUNCATE TABLE catalog_attribute');
        \DB::delete('TRUNCATE TABLE catalog_attribute_item');
        // \DB::delete('TRUNCATE TABLE catalog_category');
        \DB::delete('TRUNCATE TABLE catalog_category_item');
        // \DB::delete('TRUNCATE TABLE catalog_menu');
        // \DB::delete('TRUNCATE TABLE catalog_menu_item');
        \DB::delete('TRUNCATE TABLE catalog_item');
        \DB::delete('TRUNCATE TABLE catalog_item_option');
        \DB::delete('TRUNCATE TABLE catalog_favorite');
        \DB::delete('TRUNCATE TABLE catalog_cart_item');
        \DB::statement('SET FOREIGN_KEY_CHECKS = 1');

        // \DB::statement('ALTER TABLE `catalog_menu_item` AUTO_INCREMENT = 1');
        // \DB::statement('ALTER TABLE `catalog_category` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_category_item` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item_option` AUTO_INCREMENT = 1');
        \DB::statement('ALTER TABLE `catalog_item` AUTO_INCREMENT = 100');
    }

    /**
     * Create the association to the attribute.
     *
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     * @return AttributeItem
     */
    protected function createAttributeItem($simple, $attribute, $index)
    {
        $amount = $simple->vendor_meta['amount'] > 0 ? $simple->vendor_meta['amount'] : $simple->price;
        $value = isset($simple->vendor_meta['amount_label'])
            ? $simple->vendor_meta['amount_label']
            : (1 == $simple->exchange_to_usd ? '$' : '') . Format::amount($amount);

        $where = [
            'attribute_id' => $attribute->getKey(),
            'item_id' => $simple->getKey(),
            'value' => $value,
        ];

        $data = [
            'position' => $index + 1,
            'active' => 1
        ];

        return AttributeItem::updateOrCreate($where, $data);
    }

    /**
     * Create an item.
     *
     * @param  array  $row
     * @return Item
     */
    protected function createConfigurableItem(array $row)
    {
        $row['type'] = 'configurable';
        $row['msrp'] = 0;
        $row['cost'] = 0;
        $row['price'] = 0;

        return $this->createItem($row);
    }

    /**
     * Create an item.
     *
     * @param  array  $row
     * @return Item
     */
    protected function createItem(array $row)
    {
        if (! isset($row['catalog_vendor_id'])) {
            $row['catalog_vendor_id'] = $this->vendor()->getKey();
        }

        if (! isset($row['catalog_id'])) {
            $row['catalog_id'] = $this->catalog($row['catalog'])->getKey();
        }

        if (! isset($row['vendor_meta']) || empty($row['vendor_meta'])) {
            $row['vendor_meta'] = [];
        }

        if (! isset($row['short_description']) || empty($row['short_description'])) {
            $row['short_description'] = null;
        } else {
            $row['short_description'] = $this->truncateDescription(strip_tags($row['short_description']), 250);
        }

        if (isset($row['msrp']) && is_numeric($row['msrp'])) {
            $row['msrp'] = round($row['msrp'], 2);
        }
        if (isset($row['cost']) && is_numeric($row['cost'])) {
            $row['cost'] = round($row['cost'], 2);
        }
        if (isset($row['price']) && is_numeric($row['price'])) {
            $row['price'] = round($row['price'], 2);
        }

        // note: reloadable cards may not have any costs, msrp, or price.

        $pointValue = (float) config('catalog.default_point_value');
        $originalPoints = $row['points'] ?? null;
        if ('reloadable' === $row['class'] && empty($row['cost'])) {
            $row['cost'] = $pointValue;
        }

        if (!empty($row['cost']) && empty($row['price'])) {
            $denomination = $row['vendor_meta']['amount'] ?? $row['cost'];
            $markup = $this->getMarkupForDenomination($row['price_markup'], $denomination);
            $row['price'] = $this->calculatePrice($row['cost'], $markup);
        } elseif (empty($row['cost']) && !empty($row['price'])) {
            $denomination = $row['vendor_meta']['amount'] ?? $row['price'];
            $markup = $this->getMarkupForDenomination($row['price_markup'], $denomination);
            $row['cost'] = $this->calculateCost($row['price'], $markup);
        } else {
            $denomination = $row['vendor_meta']['amount'] ?? $row['cost'];
            $markup = $this->getMarkupForDenomination($row['price_markup'], $denomination);
        }

        $row['msrp'] = empty($row['msrp']) ? $row['cost'] : $row['msrp'];
        $row['price_markup'] = $markup;
        // margin = 1 - (1 / (1 + markup))
        $row['price_margin'] = ($row['price'] ? ($row['price'] - $row['cost']) / $row['price'] : 0);
        $row['point_value'] = $pointValue;
        $row['points'] = $this->calculatePoints($row['price'], $pointValue);

        $this->info("{$row['catalog']} - '{$row['name']}' - {$row['points']} points ({$row['sku']})");

        if ($row['price']) {
            $this->checkExpectedPrice($row['name'], $row['price'], $row['cost'], $markup);
            $this->checkExpectedPoints($row['name'], $originalPoints, $row['price']);
        }

        $allowed = [
            'catalog_vendor_id', 'catalog_id', 'code', 'sku', 'type', 'class', 'name',
            'locale', 'short_description', 'description', 'terms', 'disclaimer',
            'manufacturer', 'image', 'msrp', 'cost', 'price', 'exchange_to_usd',
            'price_markup', 'price_margin', 'point_value', 'points', 'points_min',
            'points_max', 'visibility', 'active', 'vendor_active', 'vendor_meta',
        ];
        $row = array_intersect_key($row, array_flip($allowed));

        return Item::updateOrCreate(['sku' => $row['sku']], $row);
    }

    /**
     * Create an item from a denomination value.
     *
     * @param  string $denomination
     * @param  array  $row
     * @return Item
     */
    protected function createItemFromDenomination(string $denominationString, array $row)
    {
        list($denominationLabel, $denominationValue) = $this->splitDenomination($denominationString);
        $denominationUsd = $this->exchangeToUsd($denominationValue, $row['exchange_to_usd'] ?? 1);

        $row['msrp'] = empty($row['msrp']) ? $denominationUsd : $row['msrp'];
        $row['cost'] = empty($row['cost']) ? $denominationUsd : $row['cost'];
        $row['sku'] = $row['sku'] . '_' . str_slug($denominationLabel, '_');
        $row['code'] = $row['code'] . '-' . str_slug($denominationLabel, '-');
        $row['type'] = 'simple';
        $row['name'] = sprintf(
            '%s %s%s',
            $row['name'],
            is_numeric($denominationLabel) && 1 == ($row['exchange_to_usd'] ?? 1) ? '$' : '- ',
            $denominationLabel
        );
        $row['short_description'] = '';
        $row['description'] = '';
        $row['terms'] = '';
        $row['disclaimer'] = '';
        $row['visibility'] = 0;
        $row['vendor_meta']['amount'] = $denominationValue;

        if ($denominationLabel != $denominationValue) {
            $row['vendor_meta']['amount_label'] = $denominationLabel;
        }

        return $this->createItem($row);
    }

    /**
     * Create the option.
     *
     * @param  Item      $item
     * @param  Item      $simple
     * @param  Attribute $attribute
     * @param  int       $index
     *
     * @return Option
     */
    protected function createOption($item, $simple, $attribute, $index)
    {
        $amount = $simple->vendor_meta['amount'] > 0 ? $simple->vendor_meta['amount'] : $simple->price;
        $value = isset($simple->vendor_meta['amount_label'])
            ? $simple->vendor_meta['amount_label']
            : (1 == $simple->exchange_to_usd ? '$' : '') . Format::amount($amount);

        $where = [
            'super_id' => $item->getKey(),
            'item_id' => $simple->getKey(),
            'attribute_id' => $attribute->getKey(),
            'value' => $value,
        ];

        $data = array_merge($where, [
            'label' => $value,
            'position' => $index + 1
        ]);

        return Option::updateOrCreate($where, $data);
    }

    /**
     * Convert the amount from USD based on some exchange rate.
     *
     * @param  float $amountUsd
     * @param  float $exchangeRate
     * @return int
     */
    protected function exchangeFromUsd(float $amountUsd, float $exchangeRate): float
    {
        return $amountUsd * $exchangeRate;
    }

    /**
     * Convert the amount to USD based on some exchange rate.
     *
     * @param  float $amount
     * @param  float $exchangeRate
     * @return float
     */
    protected function exchangeToUsd(float $amount, float $exchangeRate): float
    {
        return $amount / $exchangeRate;
    }

    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [];
    }

    /**
     * Get the category code from the category name.
     *
     * @param  string $category
     * @return string
     */
    protected function getCategoryCode(string $category): string
    {
        return str_slug(str_replace(['&', '+'], 'and', $category));
    }

    /**
     * The data from the CSV file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getCsvData(string $filename): array
    {
        if (! starts_with($filename, '/')) {
            if (file_exists(base_path($filename))) {
                $filename = base_path($filename);
            } else {
                $filename = storage_path($filename);
            }
        }

        $fileHandle = fopen($filename, 'r');
        $data = [];
        $headers = fgetcsv($fileHandle, 4096, ',');
        // remove BOM if exists
        $headers = array_map(function ($value) {
            return preg_replace('/^[\x{FEFF}]+/u', '', trim($value));
        }, $headers);

        while (($row = fgetcsv($fileHandle, 4096, ',')) !== FALSE) {
            // trim each value in row
            $row = array_map('trim', $row);
            $data[] = array_combine($headers, $row);
        }
        fclose($fileHandle);

        return $data;
    }

    /**
     * The data from the Excel file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getExcelData(string $filename): array
    {
        if (! starts_with($filename, '/')) {
            if (file_exists(base_path($filename))) {
                $filename = base_path($filename);
            } else {
                $filename = storage_path($filename);
            }
        }

        $spreadsheet = IOFactory::load($filename);
        $worksheet = $spreadsheet->getActiveSheet();
        $rows = [];
        $headers = [];
        $isHeader = true;
        foreach ($worksheet->getRowIterator() as $row) {
            $cellIterator = $row->getCellIterator();
            $cellIterator->setIterateOnlyExistingCells(false);
            $cells = [];
            foreach ($cellIterator as $cell) {
                // remove carriage return from excel cell value
                $value = preg_replace('/_x000D_/', "\n", $cell->getCalculatedValue() ?: '');
                $cells[] = trim($value);
            }

            if ($isHeader) {
                $isHeader = false;
                $headers = $cells;
            } else {
                $rows[] = array_combine($headers, $cells);
            }
        }

        return $rows;
    }

    /**
     * The data from the JSON file.
     *
     * @param  string $filename
     * @return array
     */
    protected function getJsonData(string $filename): array
    {
        if (! starts_with($filename, '/')) {
            if (file_exists(base_path($filename))) {
                $filename = base_path($filename);
            } else {
                $filename = storage_path($filename);
            }
        }

        $json = file_get_contents($filename);
        $data = json_decode($json, true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new \DomainException("Unable to parse json in $filename. Error: " . json_last_error_msg());
        }

        if (! isset($data['data']) || ! is_array($data['data'])) {
            throw new \DomainException("Unable to find expected top-level key named 'data'.");
        }

        return $data['data'];
    }

    /**
     * Get the markup for the given denomination.
     *
     * @param  string $markup
     * @param  float  $denomination
     * @return float
     */
    protected function getMarkupForDenomination(string $markup, float $denomination): float
    {
        if (is_numeric($markup)) {
            return (float) $markup;
        }

        $markup = json_decode($markup, true);
        if (isset($markup[$denomination])) {
            return $markup[$denomination];
        }

        return 0;
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        $fileDescription = "The file from which to import card data."
            . " It can be an absolute path, but if not then it looks in the base path and then the storage folder."
            . "\nSee the files in " . realpath(__DIR__ . '/data') . " for some default example that one could use.";

        return [
            'file' => ['file', 'f', InputOption::VALUE_OPTIONAL, $fileDescription],
            'use_default_price_margin' => [
                'use_default_price_margin',
                null,
                InputOption::VALUE_NONE,
                'Use default margin in the Catalog config instead of what is in the file.',
            ],
            'reset' => [
                'reset',
                'r',
                InputOption::VALUE_NONE,
                'Truncates catalog related tables; not available on production.',
            ],
            // 'dry-run' => [
            //     'dry-run',
            //     null,
            //     InputOption::VALUE_NONE,
            //     'Should we run the import or dry-run it?',
            //     false,
            // ],
        ];
    }

    /**
     * Ensure we have a denomination attribute.
     *
     * @return Attribute
     */
    protected function getOrCreateDenominationAttribute()
    {
        return Attribute::updateOrCreate(['code' => 'denomination'], [
            'label' => 'Denomination',
            'code' => 'denomination',
            'type' => 'select'
        ]);
    }

    /**
     * @param  array $rows
     * @return void
     */
    protected function introMessage(array $rows)
    {
        $this->info(sprintf('Importing %s %s.', $count = count($rows), str_plural('card', $count)));
        $this->warn('Points and price for each item will be re-calculated based on the denomination or cost, exchange rate, and markup.');
    }

    /**
     * Determine if user asked for a dry run.
     *
     * @return bool
     */
    protected function isDryRun()
    {
        if (is_null($this->isDryRun)) {
            $this->isDryRun = in_array($this->option('dry-run'), $this->dryRunPositiveValues, true);
        }
        return $this->isDryRun;
    }

    /**
     * @param  string $catalog
     * @return int
     */
    protected function menuId(string $catalog)
    {
        if (is_null($this->menuIds)) {
            $this->menuIds = Menu::all()->pluck('id', 'catalog_id')->toArray();
        }

        $catalogId = $this->catalog($catalog)->getKey();

        return $this->menuIds[$catalogId];
    }

    /**
     * Prepare the row
     *
     * @param  int    $number
     * @param  array  $row
     * @return array
     */
    protected function prepareRow(int $number, array $row): array
    {
        return $row;
    }

    /**
     * Process the items.
     *
     * @param  array $data
     * @return bool
     */
    protected function process(array $data)
    {
        $rows = collect($data);
        $categories = collect();

        // Create the 'denomination' attribute if it does not already exist.
        $denominationAttribute = $this->getOrCreateDenominationAttribute();

        // Extrapolate the source data into Catalog Item entities.
        foreach ($rows as $number => $row) {
            $row = $this->prepareRow($number, $row);

            if (isset($row['categories']) && ! empty($row['categories'])) {
                // @todo: do we want to create the category,
                // or throw an error if it is an unknown category?
                $this->categories(
                    $categories = $this->updateOrCreateCategories($row['categories'])->keyBy('code')
                );
                $this->addCategoryMenuItems($row['catalog'], $categories);
                unset($row['categories']);
            }

            $row['sku'] = strtolower(str_replace('-', '_', $row['sku']));
            $row['code'] = str_slug($row['sku'], '-');

            // Create new Catalog Item instances for denominations.
            // Note: reloadable cards may not have any costs, msrp, or price.
            $denominations = $row['denominations'] ?? null;
            if ($denominations) {
                $simples = collect();
                $denominations = is_string($denominations) ? explode(',', $denominations) : $denominations;
                $denominations = is_array($denominations) ? $denominations : [$denominations];
                foreach ($denominations as $denomination) {
                    // Calculate points based on markup/margin.
                    $denominationItem = $this->createItemFromDenomination($denomination, $row);
                    $simples->push($denominationItem);
                }
                $item = $this->createConfigurableItem($row);

                foreach ($simples as $index => $simple) {
                    // Associate the simple items to the configurable as catalog_item_options.
                    $item->options()->save(
                        $this->createOption($item, $simple, $denominationAttribute, $index)
                    );
                    // Associate the simple items to the configurable as catalog_attribute_items.
                    $item->attributes()->associate(
                        $this->createAttributeItem($simple, $denominationAttribute, $index)
                    );
                }

                $item->points_min = $simples->min('points');
                $item->points_max = $simples->max('points');
                unset($item->item_id);
                $item->save();
            } else {
                $item = $this->createItem($row);
                $item->points_min = $row['points'] ?? 0;
                $item->points_max = $row['points'] ?? 0;
                $item->save();
            }

            // Associate both the simple and configurable items to the correct categories.
            $categories->each(function ($category) use ($item) {
                $item->categories()->detach($category);
            });
            foreach ($categories as $category) {
                $item->categories()->attach($category->getKey(), [
                    'position' => $number + 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }

        $this->info('Done.');

        return true;
    }

    /**
     * Split $denomination into amount::name if we can
     *
     * @param  string $denomination
     * @return array
     */
    protected function splitDenomination(string $denominationString): array
    {
        if (preg_match('/^([0-9.]+)::(.+)$/', $denominationString, $matches)) {
            $amount = (float) $matches[1];
            $name = $matches[2];
        } else {
            $amount = (float) $denominationString;
            $name = $denominationString;
        }

        return [$name, $amount];
    }

    /**
     * Truncate the description.
     *
     * @param  string $description
     * @param  int    $length
     * @return string
     */
    protected function truncateDescription(string $description, int $length): string
    {
        if (strlen($description) > $length) {
            $description = substr(
                $description,
                0,
                strrpos(substr($description, 0, $length), ' ')
            ) . '...';
        }

        return $description;
    }

    /**
     * Update or create categories.
     *
     * @param  string $categories
     * @return \Illuminate\Support\Collection
     */
    protected function updateOrCreateCategories(string $categories): Collection
    {
        return collect(explode('|', $categories))->map(function ($category) {
            $code = $this->getCategoryCode($category);
            return Category::updateOrCreate(['code' => $code], [
                'code' => $code,
                'name' => $category,
                'active' => 1
            ]);
        });
    }
}
