<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Exception;
use GuzzleHttp\Exception\ClientException;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Events\IncompleteHawkOrder;
use Ignite\Catalog\Jobs\Hawk\ProcessManager;
use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

abstract class Base
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * @var Client
     */
    protected $client;

    /**
     * Send the request to hawk.
     *
     * @param $payload
     * @param string $requestId
     * @param Client $hawkApi
     *
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    abstract protected function request($payload, $requestId, Client $hawkApi);

    /**
     * The payload for the current request.
     *
     * @return Request
     */
    abstract protected function payload();

    /**
     * The expected response for the current request.
     *
     * @return string|array
     */
    abstract protected function expectedResponses();

    /**
     * Create a new job instance.
     *
     * @param \Ignite\Catalog\Entities\Order     $order
     * @param \Ignite\Catalog\Entities\OrderItem $orderItem
     */
    public function __construct(Order $order, OrderItem $orderItem)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
    }

    /**
     * Execute the job.
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle()
    {
        $requestId = $this->buildRequestId();
        $expectedResponses = Arr::wrap($this->expectedResponses());
        $manager = new ProcessManager($this->order, $this->orderItem, $requestId);

        try {
            $payload = $this->payload();
            $meta = ['request' => get_class($payload), 'payload' => $payload->toArray()];
            $response = $this->request($payload, $requestId, $this->getClient());

            if ($response instanceof ErrorResponse) {
                return $manager->error($response, $meta);
            }

            if (in_array(get_class($response), $expectedResponses)) {
                if ($manager->isOrderComplete($response)) {
                    $this->completeResponse($response, $requestId);
                    return $manager->complete($response, $meta);
                }

                $this->inCompleteResponse($response, $requestId);
            }

            return $manager->attempt($response, $meta);
        } catch (Exception $e) {
            return $manager->error(
                $this->exceptionToErrorResponse($e, $payload ?? null, $requestId),
                $meta ?? []
            );
        }
    }

    /**
     * Build up the request id.
     *
     * @return string
     */
    protected function buildRequestId()
    {
        return sprintf(
            '%s_%s_%s',
            $this->order->number,
            $this->orderItem->id,
            $this->orderItem->created_at->format('Y_m_d_H_i_s')
        );
    }

    /**
     * Run some processing on complete.
     *
     * @param Response $response
     * @param string $requestId
     */
    protected function completeResponse($response, $requestId)
    {
        return;
    }

    /**
     * Transform an exception into an error response.
     *
     * @param Exception $e
     * @param $payload
     * @param string $requestId
     * @return ErrorResponse
     */
    protected function exceptionToErrorResponse(Exception $e, $payload, string $requestId): ErrorResponse
    {
        $exceptionData = [
            'requestId' => $requestId,
            'payload' => $payload instanceof Arrayable
                ? json_encode($payload->toArray(), JSON_PRETTY_PRINT) : $payload,
            'message' => $e->getMessage(),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'code' => $e->getCode(),
            'trace' => $e->getTraceAsString(),
        ];

        if ($e instanceof ClientException) {
            $content = $e->getResponse()->getBody()->getContents();
            $e->getResponse()->getBody()->rewind();

            $exceptionData['message'] = $content;
            $exceptionData['request']['method'] = $e->getRequest()->getMethod();
            $exceptionData['request']['uri'] = $e->getRequest()->getUri();
            $exceptionData['response']['statusCode'] = $e->getResponse()->getStatusCode();
            $exceptionData['response']['reasonPhrase'] = $e->getResponse()->getReasonPhrase();

            $json = json_decode($content, true);
            if (null !== $json) {
                $exceptionData = array_merge($exceptionData, $json);
            }
        }

        return new ErrorResponse(['exception' => $exceptionData]);
    }

    /**
     * @return Client
     */
    protected function getClient(): Client
    {
        if (empty($this->client)) {
            $this->client = resolve(Client::class);
        }

        return $this->client;
    }

    /**
     * The class key can be used to lookup config values.
     *
     * @return string
     */
    protected function getClassKey(): string
    {
        $className = $this->orderItem->class;

        if (empty($className)) {
            $className = trim(preg_replace("([A-Z])", " $0", class_basename(static::class)));
            $className = Str::lower(str_replace(' ', '.', $className));
        }

        return $className;
    }

    /**
     * Format the country.
     *
     * @param null $country
     *
     * @return string|null
     */
    protected function getCountry($country = null)
    {
        if (! is_null($country)) {
            $country = Str::upper($country);
        } else {
            $country = Str::upper($this->order->ship_country);
        }

        if ($country === 'US') {
            return 'USA';
        }

        if ($country === 'CA') {
            return 'CAN';
        }

        throw new \DomainException("Unsupported country: {$country}");
    }

    /**
     * Determine the financialAccountId to use, also known as FAID.
     *
     * @return null|string
     */
    protected function getFinancialAccountId(): ?string
    {
        if (isset($this->orderItem->item->vendor_meta['financialAccountId'])) {
            return $this->orderItem->item->vendor_meta['financialAccountId'];
        }

        $key = $this->getClassKey();

        return config("catalog.vendors.hawk.{$key}.financialAccountId", null);
    }

    /**
     * @param  string $configPath
     *
     * @return string
     */
    protected function getKeyFromConfig(string $configPath): string
    {
        $keyValues = explode('.', $configPath);

        if (!is_array($keyValues)) {
            return $keyValues;
        }

        $lastKeyInArray = array_pop($keyValues);
        unset($keyValues[$lastKeyInArray]);
        $key =  implode('.', $keyValues);
        $productId = config("catalog.vendors.hawk.{$key}.test_product_id");

        if (!$productId) {
            $key = explode('.', $key);
            $lastKeyInArray = array_pop($key);
            unset($key[$lastKeyInArray]);
            $key = is_array($key) ? implode('.', $key) : $key;
            $productId = config("catalog.vendors.hawk.{$key}.test_product_id");
        }

        return $productId;
    }

    /**
     * Determine the payment type to use.
     *
     * @return string
     */
    protected function getPaymentType()
    {
        if (is_array($this->orderItem->item->vendor_meta) &&
            isset($this->orderItem->item->vendor_meta['payment_type']) &&
            (
                in_array($this->orderItem->item->vendor_meta['payment_type'], PaymentType::getPaymentTypes()) ||
                in_array($this->orderItem->item->vendor_meta['payment_type'], array_keys(PaymentType::getPaymentTypes()))
            )
        ) {
            return $this->orderItem->item->vendor_meta['payment_type'];
        }

        $key = $this->getClassKey();

        return config("catalog.vendors.hawk.{$key}.payment_type", PaymentType::ACH_DEBIT);
    }

    /**
     * Get the product ID for the current environment.
     *
     * @return string
     */
    protected function getProductId(): string
    {
        if (config('catalog.vendors.hawk.test_mode')) {
            $className = $this->getClassKey();
            $productId = config("catalog.vendors.hawk.{$className}.test_product_id");
            if (!$productId) {
                $productId = $this->getKeyFromConfig($className);
            }

            return $productId;
        } else {
            return $this->orderItem->item->vendor_meta['product_id'];
        }
    }

    /**
     * Customize the job action based on the response.
     *
     * @param Response $response
     * @param string $requestId
     *
     * @return void|bool
     */
    protected function inCompleteResponse($response, $requestId)
    {
        IncompleteHawkOrder::dispatch($this->order, $this->orderItem, $requestId);

        return true;
    }
}
