<?php

namespace Ignite\Catalog\Models\Report;

use Ignite\Core\Entities\User;
use Ignite\Core\Models\Grid\QueryTable;
use Illuminate\Support\Facades\DB;

class OrderReport extends QueryTable
{
    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [
        'username', 'company'
    ];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [
        'user_id'
    ];

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        return [
            'user_id' => [
                'title' => 'User ID',
                'name' => 'participant.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => 'Order Date',
                'name' => 'catalog_order.created_at',
                'orderable' => true,
                'exportable' => true,
            ],
            'first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'email' => [
                'title' => 'E-mail',
                'name' => 'participant.email',
                'orderable' => true,
                'exportable' => true,
            ],
            'ship_email' => [
                'title' => 'Recipient Email',
                'name' => 'catalog_order.ship_email',
                'orderable' => true,
                'exportable' => true,
            ],
            'company' => [
                'title' => 'Company',
                'name' => 'participant.company',
                'orderable' => true,
                'exportable' => true,
            ],
            'number' => [
                'title' => 'Order #',
                'name' => 'catalog_order.number',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor_order_numbers' => [
                'title' => 'Vendor Order #',
                'name' => 'item_vendor_order_numbers.vendor_order_numbers',
                'orderable' => true,
                'exportable' => true,
            ],
            'quantity' => [
                'title' => 'Quantity',
                'name' => 'catalog_order.quantity',
                'orderable' => true,
                'exportable' => true,
            ],
            // Note: catalog_order.points is already the total
            'points' => [
                'title' => 'Points',
                'name' => 'catalog_order.points',
                'orderable' => true,
                'exportable' => true,
            ],
            // 'dollar_value' => [
            //     'title' => 'Dollar Value',
            //     'name' => 'catalog_order_derived.dollar_value',
            //     'orderable' => true,
            //     'searchable' => true,
            //     'exportable' => true,
            // ],

            // @deprecated - Do not use total; it is misleading for catalog_order.
            // -- Because catalog_order can contain several line items, each with
            // -- their own quantity, the catalog_order.points is already the
            // -- total, and so does not need to multiply by quantity again.
            // 'total' => [
            //     'title' => 'Total',
            //     'name' => 'catalog_order_derived.total',
            //     'orderable' => true,
            //     'searchable' => true,
            //     'exportable' => true,
            // ],
        ];
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = $this->getColumnNames()->toArray();

        $pointValue = config('catalog.default_point_value');
        $orderSubquery = "(SELECT `id`, (`points` * {$pointValue}) AS `dollar_value` FROM `catalog_order`)"
            . " AS `catalog_order_derived`";
        $orderNumbersSubquery = "(SELECT `catalog_order_id`,"
            . " GROUP_CONCAT(DISTINCT `vendor_order_number`) AS `vendor_order_numbers`"
            . " FROM `catalog_order_item` GROUP BY `catalog_order_id`) AS `item_vendor_order_numbers`";

        return $this->getConnection()
            ->table('catalog_order')
            ->select($columns)
            ->leftJoin('core_participant as participant', 'participant.user_id', '=', 'catalog_order.user_id')
            ->leftJoin('core_user as user', 'user.user_id', '=', 'participant.user_id')
            ->leftJoin(DB::raw($orderSubquery), 'catalog_order.id', '=', 'catalog_order_derived.id')
            ->leftJoin(
                DB::raw($orderNumbersSubquery),
                'catalog_order.id',
                '=',
                'item_vendor_order_numbers.catalog_order_id'
            )
            ->where('user.internal', User::TYPE_PARTICIPANT);
    }

    /**
     * Get default builder parameters.
     *
     * @param array $params
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        $columns = array_flip(array_keys($this->getColumns()));

        $params = parent::getBuilderParameters();
        $params['order'] = [
            [$columns['created_at'], 'desc'],
        ];

        return $params;
    }
}
