<?php

namespace Ignite\Catalog\Emails;

use Ignite\Catalog\Entities\Order;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;

class OrderConfirmation extends Mailable
{
    use Queueable;
    use SerializesModels;

    /**
     * @var Order
     */
    protected $order;

    /**
     * Create a new order submitted message instance.
     *
     * @param Order $order
     */
    public function __construct($order)
    {
        $this->order = $order;
    }

    /**
     * Build the message.
     *
     * @return $this
     */
    public function build()
    {
        if (! $this->order->relationLoaded('items')) {
            $this->order->load('items');
        }

        if (! $this->order->relationLoaded('participant')) {
            $this->order->load('participant');
        }

        $data = [
            'first_name' => $this->order->participant->first,
            'last_name' => $this->order->participant->last,
            'email' => $this->order->participant->email,
            'order' => $this->order,
            'orderUrl' => $this->getOrderUrl(),
        ];

        $this->to($this->order->participant->email, $this->order->participant->fullName());
        $this->subject($this->getSubject());

        $this->from(config('mail.from.address'), config('mail.from.name'));

        if (config('catalog.orders.confirmation_bcc.enabled')) {
            $this->bcc(config('catalog.orders.confirmation_bcc.address'));
        }

        return $this->markdown('emails.orders.submitted', $data);
    }

    /**
     * The tracked URL for the order detail page.
     *
     * @return string
     */
    protected function getOrderUrl()
    {
        return route('catalog.order.show', [
            'number' => $this->order->number,
            'utm_source' => 'notification',
            'utm_medium' => 'email',
            'utm_campaign' => 'order_submitted'
        ]);
    }

    /**
     * Gets the subject for the email.
     *
     * @return string
     */
    protected function getSubject(): string
    {
        // @todo: Make subject configurable.
        // Right now, it should be easily overridden in a child class.
        return "Order Confirmation #{$this->order->number}";
    }
}
