<?php

namespace Ignite\Catalog\Tests\Unit\Validation;

use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Validation\ValidationStrategyFactory;
use Ignite\Catalog\Validation\Strategies\AbstractValidationStrategy;
use Ignite\Catalog\Validation\Strategies\DefaultValidationStrategy;
use Ignite\Catalog\Tests\TestCase;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class ValidationStrategyFactoryTest extends TestCase
{
    /**
     * @var ValidationStrategyFactory
     */
    private ValidationStrategyFactory $factory;

    /**
     */
    public function setUp() : void
    {
        parent::setUp();
        $this->factory = new ValidationStrategyFactory();
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_default_strategy_when_vendor_is_default()
    {
        $strategy = $this->factory->create('default');

        $this->assertInstanceOf(DefaultValidationStrategy::class, $strategy);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_default_strategy_when_no_custom_strategy_supports_vendor()
    {
        $strategy = $this->factory->create('unsupported_vendor');

        $this->assertInstanceOf(DefaultValidationStrategy::class, $strategy);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_custom_strategy_when_vendor_is_supported()
    {
        $customStrategy = \Mockery::mock(AbstractValidationStrategy::class);
        $customStrategy->shouldReceive('supports')
            ->with('test_vendor')
            ->andReturn(true);

        $this->factory->register($customStrategy);

        $strategy = $this->factory->create('test_vendor');

        $this->assertSame($customStrategy, $strategy);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_first_matching_strategy_when_multiple_strategies_support_vendor()
    {
        $firstStrategy = \Mockery::mock(AbstractValidationStrategy::class);
        $firstStrategy->shouldReceive('supports')
            ->with('test_vendor')
            ->andReturn(true);

        $secondStrategy = \Mockery::mock(AbstractValidationStrategy::class);
        $secondStrategy->shouldReceive('supports')
            ->with('test_vendor')
            ->andReturn(true);

        $this->factory->register($secondStrategy);
        $this->factory->register($firstStrategy);

        $strategy = $this->factory->create('test_vendor');

        $this->assertSame($firstStrategy, $strategy);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_logs_warning_when_unsupported_vendor_is_used()
    {
        Log::shouldReceive('warning')
            ->once()
            ->with('Unsupported vendor for validation strategy', \Mockery::type('array'));

        $this->factory->create('unsupported_vendor');
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_determines_vendor_name_from_single_vendor_cart()
    {
        $cart = $this->createMockCart([
            $this->createMockCartItem('vendor1'),
            $this->createMockCartItem('vendor1'),
        ]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('vendor1', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_determines_vendor_name_from_cart_with_default_vendor()
    {
        $cart = $this->createMockCart([
            $this->createMockCartItem('default'),
            $this->createMockCartItem('default'),
        ]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_determines_vendor_name_from_cart_with_null_vendor()
    {
        $cart = $this->createMockCart([
            $this->createMockCartItem(null),
            $this->createMockCartItem(null),
        ]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_default_when_cart_has_mixed_vendors()
    {
        $cart = $this->createMockCart([
            $this->createMockCartItem('vendor1'),
            $this->createMockCartItem('vendor2'),
        ]);

        Log::shouldReceive('error')
            ->once()
            ->with('Checkout contains items from multiple vendors', \Mockery::type('array'));

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_default_when_cart_is_empty()
    {
        $cart = $this->createMockCart([]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_logs_error_with_cart_and_user_info_when_mixed_vendors()
    {
        $cart = $this->createMockCart([
            $this->createMockCartItem('vendor1'),
            $this->createMockCartItem('vendor2'),
        ], 123, 456);

        Log::shouldReceive('error')
            ->once()
            ->with('Checkout contains items from multiple vendors', [
                'vendors' => ['vendor1', 'vendor2'],
                'cart_id' => 123,
                'user_id' => 456
            ]);

        ValidationStrategyFactory::determineVendorName($cart);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_cart_items_without_vendor_property()
    {
        $cartItem1 = \Mockery::mock();
        $cartItem1->vendor = null;

        $cartItem2 = \Mockery::mock();
        $cartItem2->vendor = (object)['name' => 'vendor1'];

        $cart = $this->createMockCart([$cartItem1, $cartItem2]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_cart_items_with_vendor_but_no_name()
    {
        $cartItem1 = \Mockery::mock();
        $cartItem1->vendor = (object)['name' => null];

        $cartItem2 = \Mockery::mock();
        $cartItem2->vendor = (object)['name' => 'vendor1'];

        $cart = $this->createMockCart([$cartItem1, $cartItem2]);

        $vendorName = ValidationStrategyFactory::determineVendorName($cart);

        $this->assertEquals('default', $vendorName);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_registers_custom_strategy_at_beginning_of_array()
    {
        $firstStrategy = \Mockery::mock(AbstractValidationStrategy::class);
        $secondStrategy = \Mockery::mock(AbstractValidationStrategy::class);
        $thirdStrategy = \Mockery::mock(AbstractValidationStrategy::class);

        $this->factory->register($firstStrategy);
        $this->factory->register($secondStrategy);
        $this->factory->register($thirdStrategy);

        // Use reflection to check the order
        $reflection = new \ReflectionClass($this->factory);
        $strategiesProperty = $reflection->getProperty('strategies');
        $strategiesProperty->setAccessible(true);
        $strategies = $strategiesProperty->getValue($this->factory);

        $this->assertSame($thirdStrategy, $strategies[0]);
        $this->assertSame($secondStrategy, $strategies[1]);
        $this->assertSame($firstStrategy, $strategies[2]);
    }

    /**
     * Create a mock cart with the given items
     *
     * @param array $items
     * @param int|null $cartId
     * @param int|null $userId
     * @return Cart
     */
    private function createMockCart(array $items, $cartId = null, $userId = null): Cart
    {
        $cart = \Mockery::mock(Cart::class);
        $cart->shouldReceive('items')->andReturn(collect($items));

        if ($cartId !== null) {
            $cart->id = $cartId;
        }

        if ($userId !== null) {
            // Mock Auth facade
            Auth::shouldReceive('id')->andReturn($userId);
        }

        return $cart;
    }

    /**
     * Create a mock cart item with the given vendor name
     *
     * @param string|null $vendorName
     * @return object
     */
    private function createMockCartItem($vendorName): object
    {
        $cartItem = \Mockery::mock();
        $cartItem->vendor = $vendorName ? (object)['name' => $vendorName] : null;

        return $cartItem;
    }
}
