<?php

namespace Ignite\Catalog\Tests\Unit\Validation\Strategies;

use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Validation\Strategies\HawkValidationStrategy;
use Ignite\Catalog\Tests\TestCase;
use Mockery;

class HawkValidationStrategyTest extends TestCase
{
    /**
     * @var HawkValidationStrategy
     */
    private HawkValidationStrategy $strategy;

    /**
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->strategy = new HawkValidationStrategy();
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_supports_only_hawk_vendor()
    {
        $this->assertTrue($this->strategy->supports('hawk'));
        $this->assertTrue($this->strategy->supports('HAWK'));
        $this->assertTrue($this->strategy->supports('Hawk'));
        $this->assertFalse($this->strategy->supports('default'));
        $this->assertFalse($this->strategy->supports('any-other-vendor'));
        $this->assertFalse($this->strategy->supports(''));
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_merges_base_and_hawk_rules_when_cart_is_null()
    {
        // Mock the config to return base rules and Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string', 'max:255'],
                'last' => ['required', 'string', 'max:255'],
                'ship_email' => ['required', 'email', 'max:255'],
                'ship_address_1' => ['required', 'string', 'max:255'],
                'ship_city' => ['required', 'string', 'max:255'],
                'ship_state' => ['required', 'string', 'max:255'],
                'ship_postal' => ['required', 'string', 'max:255'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
                'ship_state' => ['max:35'],
                'ship_postal' => ['max:35'],
            ]
        ]);

        $rules = $this->strategy->getValidationRules(null);

        $this->assertEquals([
            'first' => ['required', 'string', 'max:255'],
            'last' => ['required', 'string', 'max:255'],
            'ship_email' => ['required', 'email', 'max:255'],
            'ship_address_1' => ['required', 'string', 'max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            'ship_city' => ['required', 'string', 'max:35'],
            'ship_state' => ['required', 'string', 'max:35'],
            'ship_postal' => ['required', 'string', 'max:35'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_merges_base_and_hawk_rules_when_cart_has_mixed_items()
    {
        // Mock the config to return base rules and Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
                'ship_city' => ['required', 'string'],
                'ship_state' => ['required', 'string'],
                'ship_postal' => ['required', 'string'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
                'ship_state' => ['max:35'],
                'ship_postal' => ['max:35'],
            ]
        ]);

        // Create a mock cart that has both physical and non-physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should return all rules merged with Hawk-specific constraints
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['required', 'email'],
            'ship_address_1' => ['required', 'string', 'max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            'ship_city' => ['required', 'string', 'max:35'],
            'ship_state' => ['required', 'string', 'max:35'],
            'ship_postal' => ['required', 'string', 'max:35'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_filters_rules_for_cart_with_only_physical_items()
    {
        // Mock the config to return base rules and Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
                'ship_city' => ['required', 'string'],
                'ship_state' => ['required', 'string'],
                'ship_postal' => ['required', 'string'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
                'ship_state' => ['max:35'],
                'ship_postal' => ['max:35'],
            ]
        ]);

        // Create a mock cart that has only physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(true);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should remove 'required' from ship_email for physical-only carts, but keep Hawk-specific rules
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['email'], // 'required' removed
            'ship_address_1' => ['required', 'string', 'max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            'ship_city' => ['required', 'string', 'max:35'],
            'ship_state' => ['required', 'string', 'max:35'],
            'ship_postal' => ['required', 'string', 'max:35'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_filters_rules_for_cart_with_only_non_physical_items()
    {
        // Mock the config to return base rules and Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
                'ship_city' => ['required', 'string'],
                'ship_state' => ['required', 'string'],
                'ship_postal' => ['required', 'string'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
                'ship_state' => ['max:35'],
                'ship_postal' => ['max:35'],
            ]
        ]);

        // Create a mock cart that has only non-physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(true);

        $rules = $this->strategy->getValidationRules($cart);

        // Should remove 'required' from shipping address fields for non-physical-only carts, but keep Hawk-specific rules
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['required', 'email'],
            'ship_address_1' => ['string', 'max:35', 'regex:/^[a-zA-Z0-9 ]+$/'], // 'required' removed
            'ship_city' => ['string', 'max:35'], // 'required' removed
            'ship_state' => ['string', 'max:35'], // 'required' removed
            'ship_postal' => ['string', 'max:35'], // 'required' removed
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_empty_config_values()
    {
        // Mock empty config
        config(['catalog.validations.base.rules' => []]);
        config(['catalog.validations.base.messages' => []]);
        config(['catalog.validations.strategies.hawk.rules' => []]);
        config(['catalog.validations.strategies.hawk.messages' => []]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);
        $messages = $this->strategy->getValidationMessages($cart);

        $this->assertEquals([], $rules);
        $this->assertEquals([], $messages);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_context_parameter_in_get_validation_rules()
    {
        // Mock the config to return base rules and Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            ]
        ]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $context = ['some' => 'context', 'data' => 'here'];

        $rules = $this->strategy->getValidationRules($cart, $context);

        // Context should be ignored by Hawk strategy, but method should still work
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_hawk_rules_without_base_rules()
    {
        // Mock empty base rules but with Hawk-specific rules
        config([
            'catalog.validations.base.rules' => [],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
                'ship_state' => ['max:35'],
                'ship_postal' => ['max:35'],
            ]
        ]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should return only Hawk-specific rules
        $this->assertEquals([
            'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            'ship_city' => ['max:35'],
            'ship_state' => ['max:35'],
            'ship_postal' => ['max:35'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_duplicate_rules_merging()
    {
        // Mock config with overlapping rules
        config([
            'catalog.validations.base.rules' => [
                'ship_address_1' => ['required', 'string', 'max:255'],
                'ship_city' => ['required', 'string', 'max:255'],
            ],
            'catalog.validations.strategies.hawk.rules' => [
                'ship_address_1' => ['max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
                'ship_city' => ['max:35'],
            ]
        ]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should merge rules and avoid duplicates
        $this->assertEquals([
            'ship_address_1' => ['required', 'string', 'max:35', 'regex:/^[a-zA-Z0-9 ]+$/'],
            'ship_city' => ['required', 'string', 'max:35'],
        ], $rules);
    }
} 
