<?php

namespace Ignite\Catalog\Tests\Unit\Validation\Strategies;

use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Validation\Strategies\DefaultValidationStrategy;
use Ignite\Catalog\Tests\TestCase;
use Mockery;

class DefaultValidationStrategyTest extends TestCase
{
    /**
     * @var DefaultValidationStrategy
     */
    private DefaultValidationStrategy $strategy;

    /**
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->strategy = new DefaultValidationStrategy();
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_supports_all_vendors()
    {
        $this->assertTrue($this->strategy->supports('any-vendor'));
        $this->assertTrue($this->strategy->supports('hawk'));
        $this->assertTrue($this->strategy->supports('default'));
        $this->assertTrue($this->strategy->supports(''));
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_base_validation_rules_when_cart_is_null()
    {
        // Mock the config to return base rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
            ]
        ]);

        $rules = $this->strategy->getValidationRules(null);

        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['required', 'email'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_returns_base_validation_rules_when_cart_has_mixed_items()
    {
        // Mock the config to return base rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
            ]
        ]);

        // Create a mock cart that has both physical and non-physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should return all rules unchanged
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['required', 'email'],
            'ship_address_1' => ['required', 'string'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_filters_rules_for_cart_with_only_physical_items()
    {
        // Mock the config to return base rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
            ]
        ]);

        // Create a mock cart that has only physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(true);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should remove 'required' from ship_email for physical-only carts
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['email'], // 'required' removed
            'ship_address_1' => ['required', 'string'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_filters_rules_for_cart_with_only_non_physical_items()
    {
        // Mock the config to return base rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
                'ship_email' => ['required', 'email'],
                'ship_address_1' => ['required', 'string'],
                'ship_address_2' => ['required', 'string'],
                'ship_city' => ['required', 'string'],
                'ship_state' => ['required', 'string'],
                'ship_postal' => ['required', 'string'],
                'ship_country' => ['required', 'string'],
            ]
        ]);

        // Create a mock cart that has only non-physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(true);

        $rules = $this->strategy->getValidationRules($cart);

        // Should remove 'required' from shipping address fields for non-physical-only carts
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
            'ship_email' => ['required', 'email'],
            'ship_address_1' => ['string'], // 'required' removed
            'ship_address_2' => ['string'], // 'required' removed
            'ship_city' => ['string'], // 'required' removed
            'ship_state' => ['string'], // 'required' removed
            'ship_postal' => ['string'], // 'required' removed
            'ship_country' => ['string'], // 'required' removed
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_empty_config_values()
    {
        // Mock empty config
        config(['catalog.validations.base.rules' => []]);
        config(['catalog.validations.base.messages' => []]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);
        $messages = $this->strategy->getValidationMessages($cart);

        $this->assertEquals([], $rules);
        $this->assertEquals([], $messages);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_context_parameter_in_get_validation_rules()
    {
        // Mock the config to return base rules
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string'],
                'last' => ['required', 'string'],
            ]
        ]);

        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(false);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $context = ['some' => 'context', 'data' => 'here'];

        $rules = $this->strategy->getValidationRules($cart, $context);

        // Context should be ignored by default strategy, but method should still work
        $this->assertEquals([
            'first' => ['required', 'string'],
            'last' => ['required', 'string'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_rules_without_required_validation()
    {
        // Mock the config to return base rules without 'required'
        config([
            'catalog.validations.base.rules' => [
                'first' => ['string', 'max:255'],
                'last' => ['string', 'max:255'],
                'ship_email' => ['email', 'max:255'],
                'ship_address_1' => ['string', 'max:255'],
            ]
        ]);

        // Create a mock cart that has only physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(true);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should return rules unchanged since there's no 'required' to remove
        $this->assertEquals([
            'first' => ['string', 'max:255'],
            'last' => ['string', 'max:255'],
            'ship_email' => ['email', 'max:255'],
            'ship_address_1' => ['string', 'max:255'],
        ], $rules);
    }

    /**
     * @test
     * @group Catalog
     * @group Validation
     * @group ValidationStrategy
     */
    public function it_handles_mixed_required_and_non_required_rules()
    {
        // Mock the config to return base rules with mixed required/non-required
        config([
            'catalog.validations.base.rules' => [
                'first' => ['required', 'string', 'max:255'],
                'last' => ['required', 'string', 'max:255'],
                'ship_email' => ['required', 'email', 'max:255'],
                'ship_address_1' => ['required', 'string', 'max:255'],
            ]
        ]);

        // Create a mock cart that has only physical items
        $cart = Mockery::mock(Cart::class);
        $cart->shouldReceive('hasOnlyPhysical')->andReturn(true);
        $cart->shouldReceive('hasOnlyNonPhysical')->andReturn(false);

        $rules = $this->strategy->getValidationRules($cart);

        // Should remove only 'required' from ship_email, keep other rules
        $this->assertEquals([
            'first' => ['required', 'string', 'max:255'],
            'last' => ['required', 'string', 'max:255'],
            'ship_email' => ['email', 'max:255'], // 'required' removed
            'ship_address_1' => ['required', 'string', 'max:255'],
        ], $rules);
    }
}
