<?php

namespace Ignite\Catalog\Validation;

use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Validation\Strategies\AbstractValidationStrategy;
use Ignite\Catalog\Validation\Strategies\DefaultValidationStrategy;
use Ignite\Catalog\Validation\Strategies\HawkValidationStrategy;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

/**
 * Factory for creating validation strategies based on vendor
 */
class ValidationStrategyFactory
{
    /**
     * @var AbstractValidationStrategy[]
     */
    private $strategies = [
        'hawk' => HawkValidationStrategy::class,
        'default' => DefaultValidationStrategy::class
    ];

    /**
     * Determine the vendor name from the cart
     *
     * @param Cart $cart
     * @return string
     */
    static public function determineVendorName(Cart $cart): string
    {
        $vendors = [];

        foreach ($cart->items() as $item) {
            $vendorName = $item->vendor->name ?? 'default';
            $vendors[] = $vendorName;
        }

        // Check if all items are from the same vendor
        $uniqueVendors = array_unique($vendors);

        if (count($uniqueVendors) > 1) {
            // Log error for mixed vendors
            Log::error('Checkout contains items from multiple vendors', [
                'vendors' => $uniqueVendors,
                'cart_id' => $cart->id ?? 'unknown',
                'user_id' => Auth::id() ?? 'guest'
            ]);

            // Fallback to default strategy
            return 'default';
        }

        // Return the single vendor, or 'default' if none found
        return $uniqueVendors[0] ?? 'default';
    }

    /**
     * Get the appropriate validation strategy for the given vendor
     *
     * @param string $vendorName
     * @return AbstractValidationStrategy
     */
    public function create(string $vendorName): AbstractValidationStrategy
    {
        // If vendor is 'default', use default strategy
        if ($vendorName === 'default') {
            return new DefaultValidationStrategy();
        }
        foreach ($this->strategies as $strategyClass) {
            $strategy = new $strategyClass();
            if ($strategy->supports($vendorName)) {
                return $strategy;
            }
        }

        // Log warning for unsupported vendor
        Log::warning('Unsupported vendor for validation strategy', [
            'vendor_name' => $vendorName,
            'available_strategies' => $this->strategies,
        ]);

        // Fallback to default strategy
        return new DefaultValidationStrategy();
    }

    /**
     * Register a custom validation strategy
     *
     * @param AbstractValidationStrategy $strategy
     * @return void
     */
    public function register(AbstractValidationStrategy $strategy): void
    {
        // Add custom strategy at the beginning so it takes precedence
        array_unshift($this->strategies, $strategy);
    }
}
