<?php

namespace Ignite\Catalog\Validation\Strategies;

use Ignite\Catalog\Models\Cart;

/**
 * Abstract base class for validation strategies
 */
abstract class AbstractValidationStrategy
{
    /**
     * Fields to remove from validation if the cart has only non-physical items.
     *
     * @var array
     */
    protected $removeForOnlyNonPhysical = [
        'ship_address_1',
        'ship_address_2',
        'ship_address_3',
        'ship_city',
        'ship_state',
        'ship_postal',
        'ship_country',
    ];

    /**
     * Fields to remove from validation if the cart has only physical items.
     *
     * @var array
     */
    protected $removeForOnlyPhysical = [
        'ship_email',
    ];

    /**
     * Get validation rules for the checkout form based on vendor requirements
     *
     * @param Cart $cart
     * @param array $context Additional context data
     * @return array
     */
    abstract public function getValidationRules(Cart $cart, array $context = []): array;

    /**
     * Get custom validation messages for the vendor
     *
     * @param Cart|null $cart
     * @return array
     */
    abstract public function getValidationMessages(Cart $cart = null): array;

    /**
     * Check if this strategy applies to the given vendor
     *
     * @param string $vendorName
     * @return bool
     */
    abstract public function supports(string $vendorName): bool;

    /**
     * Merge field-specific validation rules
     *
     * @param array $baseFieldRules
     * @param array $vendorFieldRules
     * @return array
     */
    public function mergeFieldRules(array $baseFieldRules, array $vendorFieldRules): array
    {
        // Convert to arrays if they're strings
        if (is_string($baseFieldRules)) {
            $baseFieldRules = explode('|', $baseFieldRules);
        }

        if (is_string($vendorFieldRules)) {
            $vendorFieldRules = explode('|', $vendorFieldRules);
        }

        // Merge rules, avoiding duplicates
        $mergedRules = array_merge($baseFieldRules, $vendorFieldRules);

        // Remove duplicates while preserving order
        $uniqueRules = [];
        $ruleNames = [];

        foreach ($mergedRules as $rule) {
            // Get rule name without parameters
            $ruleName = explode(':', $rule)[0];
            if (!in_array($ruleName, $ruleNames)) {
                $uniqueRules[] = $rule;
                $ruleNames[] = $ruleName;
            } else {
                // else replace the existing rule
                $index = array_search($ruleName, $ruleNames);
                if ($index !== false) {
                    $uniqueRules[$index] = $rule;
                }
            }
        }

        return $uniqueRules;
    }

    /**
     * Remove messages based on cart items.
     *
     * @param array     $messages
     * @param Cart|null $cart
     * @return array
     */
    protected function filterMessagesByCart(array $messages, Cart $cart = null): array
    {
        if (!empty($cart) && $cart->hasOnlyPhysical()) {
            // remove 'required' message for ship_email
            foreach ($this->removeForOnlyPhysical as $field) {
                unset($messages[$field . '.required']);
            }
        }

        if (!empty($cart) && $cart->hasOnlyNonPhysical()) {
            // remove 'required' message for these if they exists
            foreach ($this->removeForOnlyNonPhysical as $field) {
                unset($messages[$field . '.required']);
            }
        }

        return $messages;
    }

    /**
     * Remove rules based on cart items.
     *
     * @param array     $rules
     * @param Cart|null $cart
     * @return array
     */
    protected function filterRulesByCart(array $rules, Cart $cart = null): array
    {
        if (!empty($cart) && $cart->hasOnlyPhysical()) {
            // remove 'required' rules for ship_email
            foreach ($this->removeForOnlyPhysical as $field) {
                if (isset($rules[$field]) && in_array('required', $rules[$field])) {
                    $rules[$field] = array_diff($rules[$field], ['required']);
                    $rules[$field] = array_values($rules[$field]);
                }
            }
        }

        if (!empty($cart) && $cart->hasOnlyNonPhysical()) {
            // remove 'required' rules for these if they exists
            foreach ($this->removeForOnlyNonPhysical as $field) {
                if (isset($rules[$field]) && in_array('required', $rules[$field])) {
                    $rules[$field] = array_diff($rules[$field], ['required']);
                    $rules[$field] = array_values($rules[$field]);
                }
            }
        }

        return $rules;
    }

    /**
     * Get base validation messages
     *
     * @return array
     */
    protected function getBaseMessages(): array
    {
        return config('catalog.validations.base.messages', []);
    }

    /**
     * Get base validation rules that apply to all vendors
     *
     * @return array
     */
    protected function getBaseRules(): array
    {
        $rules = config('catalog.validations.base.rules', []);

        return $rules;
    }

    /**
     * Get vendor-specific messages from config
     *
     * @param string $vendorName
     * @return array
     */
    protected function getVendorMessages(string $vendorName): array
    {
        return config("catalog.validations.strategies.{$vendorName}.messages", []);
    }

    /**
     * Get vendor-specific rules from config
     *
     * @param string $vendorName
     * @return array
     */
    protected function getVendorRules(string $vendorName): array
    {
        return config("catalog.validations.strategies.{$vendorName}.rules", []);
    }

    /**
     * Merge vendor-specific messages with base messages
     *
     * @param array $baseMessages
     * @param array $vendorMessages
     * @return array
     */
    protected function mergeMessages(array $baseMessages, array $vendorMessages): array
    {
        return array_merge($baseMessages, $vendorMessages);
    }

    /**
     * Merge vendor-specific rules with base rules
     *
     * @param array $baseRules
     * @param array $vendorRules
     * @return array
     */
    protected function mergeRules(array $baseRules, array $vendorRules): array
    {
        // If vendor rules are empty, return base rules
        if (empty($vendorRules)) {
            return $baseRules;
        }

        // If base rules are empty, return vendor rules
        if (empty($baseRules)) {
            return $vendorRules;
        }

        // Merge rules, with vendor rules taking precedence for same field
        $mergedRules = $baseRules;

        foreach ($vendorRules as $fieldName => $vendorFieldRules) {
            if (isset($mergedRules[$fieldName])) {
                // Field exists in both, merge the rules
                $baseFieldRules = $mergedRules[$fieldName];
                $mergedFieldRules = $this->mergeFieldRules($baseFieldRules, $vendorFieldRules);
                $mergedRules[$fieldName] = $mergedFieldRules;
            } else {
                // Field only exists in vendor rules
                $mergedRules[$fieldName] = $vendorFieldRules;
            }
        }

        return $mergedRules;
    }
} 
