<?php

namespace Ignite\Catalog\Http\Requests;

use Ignite\Catalog\Models\ReloadableThreshold;
use Illuminate\Foundation\Http\FormRequest;

class CreateReloadableRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @param  ReloadableThreshold $threshold
     * @return array
     */
    public function rules(ReloadableThreshold $threshold): array
    {
        $rules = [
            'item_id' => ['required'],
            'amount' => ['required', 'integer'],
        ];

        if (!$threshold->hasCreatedReloadableCard()) {
            $rules = array_merge($rules, $this->getInfoRules());
            $rules = array_merge($rules, $this->getVendorRules($threshold));
        }

        unset($rules['info.first'], $rules['info.last']);
        return $rules;
    }

    private function getInfoRules(): array
    {
        return [
            'info.ship_phone'    => ['required'],
            'info.ship_email'    => ['nullable', 'email:rfc,filter'],
            'info.ship_address_1'=> ['required'],
            'info.ship_city'     => ['required'],
            'info.ship_state'    => ['required'],
            'info.ship_postal'   => ['required'],
        ];
    }

    private function getVendorRules(ReloadableThreshold $threshold): array
    {
        $factory = new \Ignite\Catalog\Validation\ValidationStrategyFactory();
        $vendorName = $threshold->getItem()->vendor->name;
        $strategy = $factory->create($vendorName);
        $vendorRules = $strategy->getValidationRules($threshold->getCart());

        return collect($vendorRules)
            ->mapWithKeys(fn($value, $key) => ['info.' . $key => $value])
            ->toArray();
    }

    /**
     * The customized messages.
     *
     * @return array
     */
    public function messages(): array
    {
        return [
            'amount.required' => 'Load amount is required',
            'amount.integer' => 'Load amount is a number (no commas or decimal points)',
            'info.ship_phone.required'  => 'The phone number is required',
            'info.ship_phone.numeric'  => 'The phone number must be numeric',
            'info.ship_email.email'  => 'The email address must be a valid format',
            'info.ship_address_1.required'  => 'The address line 1 is required',
            'info.ship_address_1.max' => 'The address line 1 must not exceed 35 character limit',
            'info.ship_city.required'  => 'The city is required',
            'info.ship_city.max' => 'The city must not exceed 30 character limit',
            'info.ship_state.required'  => 'The state is required',
            'info.ship_state.max' => 'The state must not exceed 3 character limit',
            'info.ship_postal.required'  => 'The postal code/zip is required',
            'info.ship_postal.max' => 'The postal code/zip must not exceed 10 character limit',
        ];
    }


}
