<?php

namespace Ignite\Catalog\Models\Grid;

use Ignite\Catalog\Entities\Order;
use Ignite\Core\Facades\Format;
use Ignite\Packages\Presenter\Exceptions\PresenterException;
use Illuminate\Support\HtmlString;
use Ignite\Core\Models\Grid\Formatter;

class OrderFormatter extends Formatter
{
    /**
     * Format a checkbox for selecting entries in the grid.
     *
     * @param  object $model
     * @return \Illuminate\Support\HtmlString
     */
    public function checkbox(object $model)
    {
        return $this->formatCheckbox($model->getKey());
    }

    /**
     * Custom column for actions.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function actions(Order $model): HtmlString
    {
        $actions = [
            [
                'click' => '',
                'class' => '',
                'icon' => 'eye-open',
                'label' => 'View',
                'text' => 'text-default',
                'type' => 'default',
                'link' => $model->present()->getAdminShowUrl()
            ],
        ];

        return $this->buttons($actions);
    }

    /**
     * Format the items column.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function items(Order $model): HtmlString
    {
        return $this->html($model->present()->getAssociatedItemNames());
    }

    /**
     * Format an order number with a link.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function number(Order $model): HtmlString
    {
        return $this->html(sprintf(
            '<a href="%s">%s</a>',
            $model->present()->getAdminShowUrl(),
            $model->present()->getNumber()
        ));
    }

    /**
     * Format an email with a link.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function email(Order $model): HtmlString
    {
        return $this->html(sprintf(
            '<a href="mailto:%1$s">%1$s</a>',
            $model->present()->getShippingEmail()
        ));
    }

    /**
     * Format a participant on an order.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function participant(Order $model): HtmlString
    {
        return $this->html(sprintf(
            '<a href="%s"><span class="fa-solid fa-user"></span> %s</a>',
            $model->present()->getAdminParticipantShowUrl(),
            $model->present()->getParticipantName()
        ));
    }

    /**
     * Format whether an order has been processed.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function processed(Order $model): HtmlString
    {
        return $this->html($model->present()->getProcessedState());
    }

    /**
     * Format whether an order has been cancelled.
     *
     * @param  Order $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function cancelled(Order $model): HtmlString
    {
        return $this->html($model->present()->getCancelledState());
    }

    /**
     * Format whether an order has been processed.
     *
     * @param  object  $model
     * @return HtmlString
     * @throws PresenterException
     */
    public function status(object $model): HtmlString
    {
        if ($model->isProcessing()) {
            $icon = $model->present()->getProcessingIcon();
        } elseif ($model->isProcessed()) {
            $icon = $model->present()->getProcessedIcon();
        } elseif ($model->isCancelled()) {
            $icon = $model->present()->getCancelledIcon();
        } else {
            $icon = trans('Core::core.common.unknown');
        }

        return $this->html($icon);
    }

    /**
     * Format the points for an order.
     *
     * @param  Order $model
     * @return HtmlString
     */
    public function points($model)
    {
        return $this->html(Format::amount($model->points));
    }
}
