<?php

namespace Ignite\Catalog\Models\Grid;

use Illuminate\Database\DatabaseManager;
use Ignite\Core\Models\Grid\QueryTable;
use Illuminate\Database\Query\Builder;
use Yajra\DataTables\DataTables;

class CatalogsTable extends QueryTable
{
    /**
     * @var VendorFormatter
     */
    protected $formatter;

    /**
     * The forms to look up for extra columns.
     * @var array
     */
    protected $forms = [];

    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        CatalogFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title'          => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            'action' => [
                'name' => 'actions',
                'title' => trans('Core::core.common.actions'),
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions'
            ],
            'id' => [
                'title' => trans('Catalog::catalog.fields.id.label'),
                'name' => 'catalog.id',
                'orderable' => true,
                'exportable' => true,
            ],
            'code' => [
                'title' => trans('Catalog::catalog.fields.code.label'),
                'name' => 'catalog.code',
                'orderable' => true,
                'exportable' => true,
            ],
            'name' => [
                'title' => trans('Catalog::catalog.fields.name.label'),
                'name' => 'catalog.name',
                'orderable' => true,
                'exportable' => true,
            ],
            'internal_name' => [
                'title' => trans('Catalog::catalog.fields.internal_name.label'),
                'name' => 'catalog.internal_name',
                'orderable' => true,
                'exportable' => true,
            ],
            'type' => [
                'title' => trans('Catalog::catalog.fields.type.label'),
                'name' => 'catalog.type',
                'orderable' => true,
                'exportable' => true,
            ],
            'vendor' => [
                'title' => trans('Catalog::catalog.fields.vendor.label'),
                'name' => 'vendor',
                'orderable' => true,
                'exportable' => true,
            ],
            'active' => [
                'title' => trans('Catalog::catalog.fields.active.label'),
                'name' => 'catalog.active',
                'orderable' => true,
                'exportable' => true,
            ],
            'countries' => [
                'title' => trans('Catalog::catalog.fields.countries.label'),
                'name' => 'catalog.countries',
                'orderable' => true,
                'exportable' => true,
            ],
            'created_at' => [
                'title' => trans('Catalog::catalog.fields.created_at.label'),
                'name' => 'catalog.created_at',
                'orderable' => true,
                'exportable' => true,
                'visible' => true,
            ],
        ];
    }

    /**
     * The catalogs query.
     *
     * @return Builder
     */
    public function query(): Builder
    {
        $columns = collect($this->getColumns())
            ->keyBy('name')
            ->except('actions', 'checkbox', 'vendor')
            ->keys()
            ->toArray();

        $query = $this->getConnection()
            ->table('catalog')
            ->select($columns)
            ->addSelect($this->getConnection()->raw('`catalog_vendor`.`name` as `vendor`'))
            ->leftJoin('catalog_vendor', 'catalog.catalog_vendor_id', '=', 'catalog_vendor.id');

        if ($this->status !== 'active') {
            $query->where('catalog.active', $this->status);
        }

        return $query;
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'checkbox' => function ($model) { return $this->formatter->checkbox($model); },
            'action' => function ($model) { return $this->formatter->actions($model); },
            'name' => function ($model) { return $this->formatter->name($model); },
            'active' => function ($model) { return $this->formatter->active($model); },
            'created_at' => function ($model) { return $this->formatter->createdAt($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        return parent::getBuilderParameters([
            "order" => [2, 'asc'], // ID
        ], $this->getCheckboxColumnParameters());
    }
}
