<?php

namespace Ignite\Catalog\Jobs\Hawk;

use Ignite\Catalog\Jobs\Hawk\Base;
use Ignite\Vendor\Hawk\Client;
use Ignite\Vendor\Hawk\EGiftIndividualOrderLine;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\VirtualRecipient;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class Egift extends Base implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * {@inheritdoc}
     */
    protected function request($payload, $requestId, Client $hawkApi)
    {
        return $hawkApi->submitEgiftIndividual($payload, $requestId);
    }

    /**
     * {@inheritdoc}
     */
    protected function payload()
    {
        return $this->buildEgiftIndividualRequest();
    }

    /**
     * {@inheritdoc}
     */
    protected function expectedResponses()
    {
        return SubmitEGiftIndividualResponse::class;
    }

    /**
     * Build the request for the eGift Individual.
     *
     * @return SubmitEGiftIndividualRequest
     */
    protected function buildEgiftIndividualRequest()
    {
        return SubmitEGiftIndividualRequest::fromAssociativeArray([
            'financialAccountId' => $this->getFinancialAccountId(),
            'paymentType' => $this->getPaymentType(),
            'returnCardNumberAndPIN' => true,
            'poNumber' => $this->order->number,
            'clientProgramNumber' => $this->getProductId(),
            'orderDetails' => $this->buildOrderLines(),
            'emailContent' => app(EmailContent::class),
            // 'thirdPartyInfo' => app(ThirdPartyInfo::class),
        ]);
    }

    /**
     * Build the order lines depending upon the value of the order item quantity.
     *
     * @return array
     */
    protected function buildOrderLines()
    {
        return array_map(function () {
            return $this->buildOrderLine();
        }, range(1, (int)$this->orderItem->quantity, 1));
    }

    /**
     * Build order line.
     *
     * @return EgiftIndividualOrderLine
     */
    protected function buildOrderLine()
    {
        return EgiftIndividualOrderLine::fromAssociativeArray([
            'recipient' => $this->getRecipientData(),
            'amount' => (int)$this->orderItem->cost,
            'contentProvider' => $this->orderItem->item->vendor_meta['content_provider_code'],
        ]);
    }

    /**
     * The recipient data from the order.
     *
     * @return VirtualRecipient
     */
    protected function getRecipientData()
    {
        $matches = preg_split('/\s/', $this->order->ship_name, 2);
        $first = $matches[0] ?? null;
        $last = $matches[1] ?? null;

        return VirtualRecipient::fromAssociativeArray([
            'firstName' => $this->onlyGetAlphaCharacters($first),
            'lastName' => $this->getCleanedLastName($last),
            'email' => $this->order->ship_email,
            'id' => $this->order->user_id
        ]);
    }

    /**
     * @param $lastName
     * @return false|string
     */
    protected function getCleanedLastName($lastName)
    {
        $lastName = $this->onlyGetAlphaCharacters($lastName);
        $wordsInLastName = explode(" ", $lastName);
        $wordsInLastName = ucwords($wordsInLastName[0]);

        return strlen($wordsInLastName) > 29 ? substr($wordsInLastName, 0, 20) : $wordsInLastName;
    }

    /**
     * only return alpha[A-Z/a-z] characters from the string.
     *
     * @param $string
     * @return string
     */
    private function onlyGetAlphaCharacters($string): string
    {
        return trim(preg_replace('/[^A-Za-z\-]/', ' ', $string));
    }
}
