<?php

namespace Ignite\Catalog\Http\Controllers\Admin;

use Ignite\Catalog\Contracts\OrderRepository;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Http\Forms\OrderItemChangeStatusForm;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Entities\Transaction;
use Ignite\Flash\Facades\Flash;
use Illuminate\Http\RedirectResponse;
use Kris\LaravelFormBuilder\FormBuilder;

class OrderItemsStatusController extends Controller
{
    /**
     * Display a form to update order item status.
     *
     * @param  OrderItem   $orderItem
     * @param  string      $status
     * @param  FormBuilder $formBuilder
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(OrderItem $orderItem, $status, FormBuilder $formBuilder)
    {
        $orderItem = $orderItem->load(['vendor']);

        $form = $formBuilder->create(OrderItemChangeStatusForm::class, [
            'method' => 'POST',
            'url' => route('admin.catalog.orders.item.status.update', [
                'orderItem' => $orderItem->getKey(),
                'status' => $status
            ]),
            'model' => $orderItem
        ], [
            'status' => $status,
            'quantity' => $orderItem->quantity
        ]);

        return view('Catalog::admin.orders.items.modal.edit', compact('orderItem', 'status', 'form'));
    }

    /**
     * Update the order item status and any ancillary data.
     *
     * @param  OrderItem       $orderItem
     * @param  string          $status
     * @param  FormBuilder     $formBuilder
     * @param  OrderRepository $orderRepository
     *
     * @return RedirectResponse
     */
    public function update(OrderItem $orderItem, $status, FormBuilder $formBuilder, OrderRepository $orderRepository)
    {
        $orderItem = $orderItem->load(['vendor']);

        $form = $formBuilder->create(OrderItemChangeStatusForm::class, [], compact('status'));

        if (! $form->isValid()) {
            return redirect()->back()->withErrors($form->getErrors())->withInput();
        }

        $values = $form->getFieldValues();

        $orderItem->addAttempt($this->buildAttempt($status, $orderItem, $values));

        if ($status === 'cancelled') {
            $orderItem->cancel($values);
            $this->returnTransaction($orderItem);
        } elseif ($status === 'processed') {
            $orderItem->process($values);
        } else {
            Flash::error("Unknown status '$status'. Nothing has changed.");
        }

        $orderItem->save();

        // $orderItem->order->quantity is not reliable because of Helix.
        // For Helix orders, sometimes it is the number of products, not the actual total quantity of all items!!
        // Rather, we should check that all items are canceled.
        $itemsCancelled = $orderItem->order->items->filter(function (OrderItem $item) {
            return ($item->quantity == $item->cancelled_quantity);
        });
        if ($orderItem->order->items->count() === $itemsCancelled->count()) {
            $orderRepository->cancel($orderItem->order->number);
        }

        return redirect()->back();
    }

    /**
     * Return points as transaction for order item.
     *
     * @param  OrderItem $orderItem
     * @return self
     */
    private function returnTransaction(OrderItem $orderItem)
    {
        Transaction::create([
            'user_id' => $orderItem->order->user_id,
            'related_id' => $orderItem->getKey(),
            'related_name' => 'ORDER ITEM',
            'related_type' => OrderItem::class,
            'description' => $orderItem->name . ' on ' . $orderItem->order->number . ' cannot be fulfilled.',
            'type' => 'RETURN',
            'value' => ($orderItem->points * $orderItem->cancelled_quantity),
            'tax_date' => now(),
            'transaction_date' => now(),
        ]);

        return $this;
    }

    /**
     *  Build an attempt.
     *
     * @todo   Create a more reliable model for attempts.
     * @param  string    $status
     * @param  OrderItem $orderItem
     * @param  array     $formValues
     * @return array
     */
    private function buildAttempt($status, $orderItem, $formValues)
    {
        return [
            'status' => $status,
            'type' => 'manual',
            'item' => [
                'product_id' => $orderItem->item->vendor_meta['product_id'] ?? 'None',
                'content_provider_code' => $orderItem->item->vendor_meta['content_provider_code'] ?? 'None'
            ],
            'response' => [
                'date' => $formValues['date'],
                'quantity' => $formValues['quantity'],
                "success" => ($status === 'processed'),
                "isCompleted" => ($status === 'processed'),
                "orderNumber" => $formValues['number'] ?? 'N/A',
                "transactionId" => "N/A when attempted manually.",
                "percentComplete" => ($status === 'processed') ? 100 : 0,
                'message' => $formValues['notes'],
            ]
        ];
    }
}
