<?php

namespace Ignite\Catalog\Entities;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Ignite\Core\Traits\Markdownable;

class Category extends Model
{
    use Markdownable;

    const STATUS_ACTIVE = 1;
    const STATUS_INACTIVE = 0;

    /** @var string */
    protected $table = 'catalog_category';

    /** @var string */
    protected $primaryKey = 'id';

    /** @var array */
    protected $guarded = [];

    /** @var array */
    protected $casts = [
        'active' => 'boolean'
    ];

    /**
     * The relationship to the catalog menu items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function menuItems()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\MenuItem::class, 'catalog_category_id', 'id');
    }

    /**
     * The relationship to the catalog menu items.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function categoryItems()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\CategoryItem::class, 'category_id', 'id');
    }

    /**
     * Scope a query to include only active records.
     *
     * @param Builder $query
     */
    public function scopeOnlyActive(Builder $query)
    {
        $query->where('active', static::STATUS_ACTIVE);
    }

    /**
     * Scope a query to include only active records.
     *
     * @param Builder $query
     */
    public function scopeOnlyInctive(Builder $query)
    {
        $query->where('active', static::STATUS_INACTIVE);
    }

    /**
     * The data to be stored in the session.
     *
     * @return array
     */
    public function toSessionData()
    {
        return [
            'id' => $this->getKey(),
            'name' => $this->name,
            'code' => $this->code
        ];
    }

    /**
     * The array of presentable data for the view.
     *
     * @return array
     */
    public function toPresentableArray()
    {
        return [
            'name' => $this->name,
            'code' => $this->code,
            'description' => $this->getDescription(),
            'image' => $this->getImageUrl(),
            'icon' => $this->icon ?? 'None',
            'active' => $this->active ? 'Yes' : 'No'
        ];
    }

    /**
     * Set the active column.
     *
     * @param mixed $active
     */
    public function setActiveAttribute($active)
    {
        $this->attributes['active'] = (int) $active;
    }

    /**
     * Format the description.
     *
     * @param  bool $plain
     * @return string
     */
    public function getDescription($plain = false)
    {
        $description = $this->description;

        return ($plain) ? $description : $this->markdown($description);
    }

    /**
     * The image url on the correct storage location.
     *
     * @return string
     */
    public function getImageUrl()
    {
        if (! $this->image) {
            return '';
        }

        $path = $this->getImagePath() . $this->image;

        // @todo Refactor so the call to Storage is hidden behind a dependency that we own
        if (Storage::disk('s3')->has($path)) {
            return Storage::disk('s3')->url($path);
        }

        return Storage::disk('ignite')->url($path);
    }

    /**
     * The image path.
     *
     * @return string
     */
    public function getImagePath()
    {
        return 'catalog/categories/';
    }


    /**
     * Attach an image to the category.
     *
     * @param  UploadedFile $image
     * @return Category
     */
    public function attachImage(UploadedFile $image)
    {
        $image->storePubliclyAs($this->getImagePath(), $image->getClientOriginalName(), [
            'disk' => config('catalog.disk')
        ]);

        $this->image = $image->getClientOriginalName();

        return $this;
    }
}
