<?php

namespace Ignite\Catalog\Jobs\Tango;

use Exception;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ServerException;
use Ignite\Catalog\Entities\Order;
use Ignite\Catalog\Entities\OrderItem;
use Ignite\Catalog\Events\IncompleteTangoOrder;
use Ignite\Catalog\Jobs\Tango\ProcessManager;
use Ignite\Vendor\Tango\Client;
use Ignite\Vendor\Tango\Response\ErrorResponse;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

abstract class Base
{
    /**
     * @var Order
     */
    protected $order;

    /**
     * @var OrderItem
     */
    protected $orderItem;

    /**
     * @var Client
     */
    protected $client;

    /**
     * Send the request to Tango.
     *
     * @param $payload
     * @param string $requestId
     *
     * @return mixed
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Tango\ResponseException
     */
    abstract protected function request($payload, $requestId);

    /**
     * The payload for the current request.
     *
     * @return Request
     */
    abstract protected function payload();

    /**
     * The expected response for the current request.
     *
     * @return string|array
     */
    abstract protected function expectedResponses();

    /**
     * Create a new job instance.
     *
     * @param \Ignite\Catalog\Entities\Order     $order
     * @param \Ignite\Catalog\Entities\OrderItem $orderItem
     */
    public function __construct(Order $order, OrderItem $orderItem)
    {
        $this->order = $order;
        $this->orderItem = $orderItem;
    }

    /**
     * Execute the job.
     *
     * @return bool
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle()
    {
        $requestId = $this->buildRequestId();
        $expectedResponses = Arr::wrap($this->expectedResponses());
        $manager = new ProcessManager($this->order, $this->orderItem, $requestId);

        try {
            $payload = $this->payload();
            $meta = ['request' => get_class($payload), 'payload' => $payload->toArray()];
            $response = $this->request($payload, $requestId);
            if ($response instanceof ErrorResponse) {
                return $manager->error($response, $meta);
            }

            if (in_array(get_class($response), $expectedResponses)) {
                if ($manager->isOrderComplete($response)) {
                    $this->completeResponse($response, $requestId);
                    return $manager->complete($response, $meta);
                }
                $this->inCompleteResponse($response, $requestId);
            }
            return $manager->attempt($response, $meta);
        } catch (Exception $e) {
            return $manager->error(
                $this->exceptionToErrorResponse($e, $payload ?? null, $requestId),
                $meta ?? []
            );
        }
    }

    /**
     * Build up the request id.
     *
     * @return string
     */
    protected function buildRequestId()
    {
        return sprintf(
            '%s_%s_%s',
            $this->order->number,
            $this->orderItem->id,
            $this->orderItem->created_at->format('Y_m_d_H_i_s')
        );
    }

    /**
     * Run some processing on complete.
     *
     * @param Response $response
     * @param string $requestId
     */
    protected function completeResponse($response, $requestId)
    {
        return;
    }

    /**
     * Transform an exception into an error response.
     *
     * @param Exception $e
     * @param $payload
     * @param string $requestId
     * @return ErrorResponse
     */
    protected function exceptionToErrorResponse(Exception $e, $payload, string $requestId): ErrorResponse
    {
        if ($e instanceof ClientException || $e instanceof ServerException) {
            $message = sprintf(
                '`%s %s` resulted in a `%s %s` response: %s',
                $e->getRequest()->getMethod(),
                $e->getRequest()->getUri(),
                $e->getResponse()->getStatusCode(),
                $e->getResponse()->getReasonPhrase(),
                $e->getResponse()->getBody()->getContents()
            );
            $e->getResponse()->getBody()->rewind();
        } else {
            $message = $e->getMessage();
        }

        return new ErrorResponse([
            'exception' => [
                'requestId' => $requestId,
                'message' => $message,
                'payload' => $payload instanceof Arrayable
                    ? json_encode($payload->toArray(), JSON_PRETTY_PRINT) : $payload,
                'code' => $e->getCode(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ]
        ]);
    }

    /**
     * @return Client
     */
    protected function getClient(): Client
    {
        if (empty($this->client)) {
            $this->client = resolve(Client::class);
        }

        return $this->client;
    }

    /**
     * The class key can be used to lookup config values.
     *
     * @return string
     */
    protected function getClassKey(): string
    {
        $className = $this->orderItem->class;

        if (empty($className)) {
            $className = trim(preg_replace("([A-Z])", " $0", class_basename(static::class)));
            $className = Str::lower(str_replace(' ', '.', $className));
        }

        return $className;
    }

    /**
     * Format the country.
     *
     * @param null $country
     *
     * @return string|null
     */
    protected function getCountry($country = null)
    {
        if (! is_null($country)) {
            $country = Str::upper($country);
        } else {
            $country = Str::upper($this->order->ship_country);
        }

        if ($country === 'US') {
            return 'USA';
        }

        if ($country === 'CA') {
            return 'CAN';
        }

        throw new \DomainException("Unsupported country: {$country}");
    }

    /**
     * Customize the job action based on the response.
     *
     * @param Response $response
     * @param string $requestId
     *
     * @return void|bool
     */
    protected function inCompleteResponse($response, $requestId)
    {
        IncompleteTangoOrder::dispatch($this->order, $this->orderItem, $requestId);

        return true;
    }
}
