<?php

namespace Ignite\Catalog\Jobs\Tango;

use Ignite\Catalog\Jobs\Tango\Base;
use Ignite\Vendor\Tango\Request\OrderRequest;
use Ignite\Vendor\Tango\Response\OrderResponse;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class Order extends Base implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * @return OrderRequest
     */
    protected function buildOrderRequest(): OrderRequest
    {
        list($first, $last) = preg_split('/\s/', $this->order->ship_name, 2);
        $itemMeta = $this->orderItem->item->vendor_meta;

        // https://developers.tangocard.com/docs/manage-orders
        $request = [
            'accountIdentifier' => $itemMeta['account_identifier'],
            'customerIdentifier' => config('catalog.vendors.tango.customer_identifier'),
            'amount' => $this->getAmount(),
            // (Optional) Campaign may be used to administratively categorize a specific Order.
            'campaign' => $itemMeta['campaign'] ?? null,
            'emailSubject' => config('catalog.vendors.tango.email_subject'),
            'etid' => $itemMeta['etid'],
            // (Optional) An idempotent field that can be used for client-side
            // Order cross reference and prevent accidental order duplication.
            'externalRefID' => $this->createOrderNumber(),
            // (Optional) The Purchase Order Number associated with this order.
            'purchaseOrderNumber' => $this->createPurchaseOrderNumber(),
            'message' => config('catalog.vendors.tango.message'),
            'notes' => $this->getNotes(),
            'sendEmail' => config('catalog.vendors.tango.send_email', true) ? 'true' : 'false',
            'utid' => $itemMeta['utid'],
            'recipient' => [
                'email' => $this->order->ship_email,
                'firstName' => $first,
                'lastName' => $last,
            ],
            'sender' => [
                'email' => config('catalog.vendors.tango.sender.email'),
                'firstName' => config('catalog.vendors.tango.sender.firstname'),
                'lastName' => config('catalog.vendors.tango.sender.lastname'),
            ],
        ];

        if (config('catalog.vendors.tango.test.enabled')
            && config('catalog.vendors.tango.test.use_test_card')
        ) {
            $request = $this->useTestCard($request);
        }

        return OrderRequest::fromAssociativeArray($request);
    }

    /**
     * @return string
     */
    protected function createOrderNumber(): string
    {
        return $this->order->number . '-' . $this->orderItem->id;
    }

    /**
     * @return string
     */
    protected function createPurchaseOrderNumber(): string
    {
        $prefix = config('catalog.order.prefix');
        $prefix = preg_replace('/[\s_-]/', '', $prefix);

        return $prefix;
    }

    /**
     * {@inheritdoc}
     */
    protected function expectedResponses()
    {
        return OrderResponse::class;
    }

    /**
     *
     * @return int
     */
    protected function getAmount(): int
    {
        return $this->orderItem->item->vendor_meta['amount']
            ?? $this->orderItem->item->cost
            ?? $this->orderItem->item->msrp;
    }

    /**
     * @return string
     */
    protected function getNotes(): string
    {
        $notes = config('catalog.vendors.tango.notes');
        // for accounting, they would like to see the order number in the invoice notes
        return $this->createOrderNumber() . ($notes ? ': ' . $notes : '');
    }

    /**
     * {@inheritdoc}
     */
    protected function payload()
    {
        return $this->buildOrderRequest();
    }

    /**
     * {@inheritdoc}
     */
    protected function request($payload, $requestId)
    {
        return $this->getClient()->postOrder($payload, $requestId);
    }

    /**
     * Use data for a test card.
     *
     * @param  array  $request
     * @return array
     */
    protected function useTestCard(array $request): array
    {
        $request['accountIdentifier'] = config('catalog.vendors.tango.test.account_identifier');
        $request['customerIdentifier'] = config('catalog.vendors.tango.test.customer_identifier');
        $request['etid'] = config('catalog.vendors.tango.test.etid');
        $request['utid'] = config('catalog.vendors.tango.test.utid');
        $request['amount'] = config('catalog.vendors.tango.test.amount');

        return $request;
    }
}
