<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Support\Collection;
use Ignite\Catalog\Entities\Vendor;

class VendorRepository implements \Ignite\Catalog\Contracts\VendorRepository
{
    /**
     * The default vendor query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Vendor::query();
    }

    /**
     * Find all active vendors.
     *
     * @return \Illuminate\Support\Collection
     */
    public function findActive()
    {
        return $this->query()->onlyActive()->get();
    }

    /**
     * Find all vendors.
     *
     * @return \Illuminate\Support\Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Find a vendor.
     *
     * @param  int $id
     * @return \Ignite\Catalog\Entities\Vendor
     */
    public function find($id)
    {
        return Vendor::findOrFail($id);
    }

    /**
     * Find all vendors.
     *
     * @return \Illuminate\Support\Collection
     */
    public function findAllPaginated($perPage = 25)
    {
        return $this->query()->paginate($perPage);
    }

    /**
     * Create a vendor.
     *
     * @param  Collection $data
     * @return Vendor
     */
    public function create(Collection $data)
    {
        $vendor = Vendor::create($data->toArray());

        return $vendor;
    }

    /**
     * Update a vendor.
     *
     * @param  int $id
     * @param  Collection $data
     * @return Vendor
     */
    public function update($id, Collection $data)
    {
        $vendor = $this->find($id);

        $vendor->update($data->toArray());

        return $vendor;
    }

    /**
     * Delete the records with the given IDs.
     *
     * @param  int $id
     * @return Vendor
     */
    public function delete($id)
    {
        $vendor = $this->find($id);

        $vendor->forceDelete();

        return $vendor;
    }

    /**
     * Delete many vendors and associated data.
     *
     * @param  array $ids
     * @return bool
     */
    public function deleteMany(array $ids)
    {
        return app('db')->transaction(function () use ($ids) {
            return Vendor::query()->whereIn('id', $ids)->delete();
        });
    }
}
