<?php

namespace Ignite\Catalog\Models\Dashboard\Charts;

use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Ignite\Core\Contracts\Dashboard\WithInlineSeries;
use Ignite\Core\Contracts\Dashboard\WithPercentageChange;
use Ignite\Core\Models\Dashboard\Concerns\DeterminesPercentageChange;
use Ignite\Core\Models\Dashboard\Stat;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\DB;

abstract class OrdersStat extends Stat implements WithInlineSeries, WithPercentageChange
{
    use DeterminesPercentageChange;

    /**
     * The value query scoped to the given range.
     *
     * @param  array $range
     * @return Builder
     */
    protected function buildRangedValue(array $range)
    {
        DB::statement("SET SESSION sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))");

        return DB::query()
            ->select(DB::raw('COUNT(id) as total'))
            ->from('catalog_order_item')
            ->whereBetween('created_at', $range);
    }

    /**
     * The series query scoped to the given range.
     *
     * @param array $range
     *
     * @return \Illuminate\Database\Query\Builder
     */
    protected function buildRangedSeries($range)
    {
        DB::statement("SET SESSION sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))");

        return DB::table('catalog_order')
                 ->selectRaw('DATE(created_at) as `date`, COUNT(id) as `total`')
                 ->groupBy('date')
                 ->havingBetween('date', $range)
                 ->orderBy('date', 'asc');
    }

    /**
     * Get the correct data value from storage.
     *
     * @return mixed
     */
    public function getData()
    {
        if (! $this->get('data', false)) {
            $current = $this->scope($this->buildRangedValue($this->getDateRange()))->count();
            $this->attributes['data'] = $current;
        }

        if ($this instanceof WithInlineSeries) {
            $this->setSeries($this->getSeriesData());
        }

        if ($this instanceof WithPercentageChange) {
            $previous = $this->scope($this->buildRangedValue($this->getPreviousDateRange()))->count();
            $this->setPercentageChange($current, $previous);
        }

        return $this->get('data');
    }

    /**
     * Query the database for series data.
     *
     * @return array
     */
    protected function getSeriesData()
    {
        $results = $this->scope($this->buildRangedSeries($this->getDateRange()))->get();

        return $results->keyBy('date')->map->total->toArray();
    }

    /**
     * @param array $data
     */
    public function setSeries(array $data)
    {
        [$start, $end] = $this->getDateRange();
        $period = CarbonPeriod::create($start, $end);

        $series = [];

        /** @var Carbon $date */
        foreach ($period as $date) {
            $lookup = $date->format('Y-m-d');
            $key = $date->format('d');

            if (! array_key_exists($lookup, $data)) {
                if ($date->gt(today())) {
                    continue;
                } else {
                    $series[$key] = 0;
                }
            } else {
                $series[$key] = $data[$lookup];
            }
        }

        $this->offsetSet('series', $series);
    }

    /**
     * Set the percentage change, the change direction and the previous range value.
     *
     * @param int|float $current
     * @param int|float $previous
     */
    public function setPercentageChange($current, $previous)
    {
        $this->offsetSet('direction', $this->determineChangeType($current, $previous));
        $this->offsetSet('percentage', $this->determinePercentageChange($current, $previous));
        $this->offsetSet('current', $current);
        $this->offsetSet('previous', $previous);
    }
}
