<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Entities\AttributeItem;
use Ignite\Catalog\Entities\CategoryItem;
use Ignite\Catalog\Entities\Favorite;
use Ignite\Catalog\Entities\Item;

class ItemRepository implements \Ignite\Catalog\Contracts\CatalogItemRepository
{
    /**
     * The base query builder.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Item::query();
    }

    /**
     * Find an Item by id.
     *
     * @param  int $id
     * @return Item
     */
    public function find($id)
    {
        return $this->query()->byId($id)->first();
    }

    /**
     * Find an Item by sku.
     *
     * @param  string $sku
     * @return Item
     */
    public function findBySku($sku)
    {
        return $this->query()->bySku($sku)->first();
    }

    /**
     * Find an Item by code.
     *
     * @param  string $code
     * @return Item
     */
    public function findByCode($code)
    {
        return $this->query()->byCode($code)->first();
    }

    /**
     * Find catalog items by catalog.
     *
     * @param  \Ignite\Catalog\Entities\Catalog $catalog
     * @return Collection
     */
    public function findByCatalog($catalog)
    {
        return $this->query()->byCatalog($catalog)->get();
    }

    /**
     * Find all catalog items.
     *
     * @return Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Paginate catalog items.
     *
     * @param  int $records
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function paginate($records = 25)
    {
        return $this->query()->paginate($records);
    }

    /**
     * Create a new catalog item.
     *
     * @param  array  $data
     * @return Item
     */
    public function create($data)
    {
        $data = collect($data);
        $item = new Item();
        $item->forceFill($data->except('image', 'categories', 'attributes')->toArray());
        $item->vendor_meta = [];
        $item->save();

        $categories = $this->extractCategories($data);
        if (! empty($categories)) {
            $item->attachCategories($categories);
        }

        $image = $this->extractImage($data);
        if ($image instanceof UploadedFile) {
            $item->attachImage($image);
        }

        $item->save();


        return $item;
    }

    /**
     * Update an existing catalog item.
     *
     * @param  int    $id
     * @param  array  $data
     * @return Item
     */
    public function update($id, $data)
    {
        $data = collect($data);
        $item = $this->find($id);

        $categories = $this->extractCategories($data);
        if (! empty($categories)) {
            $item->attachCategories($categories);
        }

        $image = $this->extractImage($data);
        if ($image instanceof UploadedFile) {
            $item->attachImage($image);
        }

        if ($item->isConfigurable()) {
            $options = $this->extractAttributes($data);
            if (! empty($options)) {
                $item->attachOptions($options);
            }
        }

        $item->vendor_meta = [];
        $item->forceFill($data->except('image', 'categories', 'attributes')->toArray());
        $item->save();

        return $item;
    }

    /**
     * Extract the attributes data from the collection.
     *
     * @param Collection $data
     * @param array $default
     * @return array|mixed
     */
    protected function extractAttributes(Collection $data, $default = [])
    {
        if (! $data->has('attributes')) {
            return $default;
        }

        $json = $data->pull('attributes');

        if (is_string($json) && '' === trim($json)) {
            return $default;
        }

        $attributes = json_decode($json, true);

        if (json_last_error() !== JSON_ERROR_NONE || empty($attributes)) {
            return $default;
        }

        return $attributes;
    }

    /**
     * Extract the categories data from the collection.
     *
     * @param  Collection $data
     * @param  array $default
     * @return array
     */
    protected function extractCategories(Collection $data, $default = [])
    {
        if (! $data->has('categories')) {
            return $default;
        }

        $categories = $data->pull('categories');

        if (empty($categories)) {
            return $default;
        }

        return $categories;
    }

    /**
     * Extract the image from the data.
     *
     * @param  Collection $data
     * @param  mixed $default
     * @return UploadedFile|mixed
     */
    protected function extractImage(Collection $data, $default = false)
    {
        if (! $data->has('image')) {
            return $default;
        }

        $image = $data->pull('image');

        if (is_null($image) || (is_string($image) && '' === trim($image))) {
            return $default;
        }

        if (! $image instanceof UploadedFile) {
            throw new \UnexpectedValueException('Unexpected value for image, please upload a file.');
        }

        return $image;
    }

    /**
     * Enable a catalog item.
     *
     * @param  int $id
     * @return bool
     */
    public function enable($id)
    {
        return $this->query()->byId($id)->update(['visibility' => 1]);
    }

    /**
     * Disable a catalog item.
     *
     * @param  int $id
     * @return bool
     */
    public function disable($id)
    {
        return $this->query()->byId($id)->update(['visibility' => 0]);
    }

    /**
     * Delete the records with the given IDs.
     *
     * @param  string|array $ids
     * @return bool
     */
    public function delete($ids)
    {
        DB::beginTransaction();

        try {
            /** @var \Illuminate\Support\Collection $items */
            $items = Item::byIds($ids)->get();

            /** @var Item $item */
            foreach ($items as $item) {
                // Check if the item belongs to any orders/order_items?
                $orderItemCount = \Ignite\Catalog\Entities\OrderItem::query()
                    ->where('catalog_item_id', $item->getKey())
                    ->count();

                if ($orderItemCount > 0) {
                    throw new \DomainException(
                        sprintf(
                            '%s is associated with %s %s and could not be deleted.',
                            $item->name,
                            $orderItemCount,
                            str_plural('order', $orderItemCount)
                        )
                    );
                }

                // if it is a configurable, delete associations
                if ($item->isConfigurable()) {
                    $item->options()->delete();
                }


                AttributeItem::where('item_id', $item->getKey())->delete();

                // Delete any associated catalog_item_option entries
                $item->associated()->delete();

                // Delete any related catalog_category_item entries
                CategoryItem::where('item_id', $item->getKey())->delete();
                //$item->categories()->delete();

                // Delete any related catalog_favorite entries
                Favorite::where('item_id', $item->getKey())->orWhere('super_id', $item->getKey())->delete();

                // Delete the item
                $item->forceDelete();
            }
        } catch (\Exception $e) {
            DB::rollBack();
            throw new \DomainException("Unable to delete records with IDs: {$ids} - {$e->getMessage()}");
        }

        DB::commit();

        return true;
    }

    /**
     * Does/should the item have a URL to go to?
     *
     * @param  Item $item
     * @return boolean
     */
    public function hasUrl(Item $item): bool
    {
        $has = ['egift', 'giftcard', 'prepaid.anonymous.bulk', 'reloadable'];
        if (in_array($item->class, $has)) {
            return true;
        }

        return false;
    }

    /**
     * The canonical item url.
     *
     * @param  array $append
     * @return null|string
     */
    public function getUrl(Item $item, $append = []): ?string
    {
        if (in_array($item->class, ['egift', 'giftcard', 'prepaid.anonymous.bulk'])) {
            return route('catalog.item.show', array_merge($append, ['item' => $item->getKey()]));
        } elseif ('reloadable' === $item->class) {
            return route('catalog.hawk-reloadable', $append);
        }

        return null;
    }

    /**
     * Is this item limited to just one quantity per line item?
     *
     * Usually this is because the API (like Tango) does not have a way to
     * order multiple quantities per line item.
     *
     * @return boolean
     */
    public function isLineItemQuantityLimitedToOne(Item $item): bool
    {
        // if Tango
        if (stripos($item->vendor->name, 'tango') !== false
            || stripos($item->vendor->redemption_class, 'tango') !== false
        ) {
            /* @todo: this could be a temporary solution
             * -------------------------------------------------- //
             * We either need to:
             *
             * 1) move this to the VendorHandler or Processor class for each
             * vendor type or
             *
             * 2) we do not need this anymore by allowing to group multiple
             * quantities together into to one line item like normal, and have
             * the processor handle it with multiple API calls.
             * -------------------------------------------------- */
            return true;
        }

        return false;
    }
}
