<?php

namespace Ignite\Catalog\Models;

use Ignite\Catalog\Entities\Item;
use Ignite\Core\Entities\Participant;

class ReloadableThreshold
{
    /**
     * An instance of the Cart model.
     *
     * @var Cart
     */
    protected $cart;

    /**
     * The date the participant created the reloadable card.
     *
     * @var null|string|bool
     */
    protected $reloadable_created_at = false;

    /**
     * Cache the check value for whether they have ordered previously.
     *
     * @var bool
     */
    protected $isFirstOrder;

    /**
     * ReloadableVerification constructor.
     *
     * @param Cart $cart
     */
    public function __construct(Cart $cart)
    {
        $this->cart = $cart;
    }

    /**
     * Verify whether the participant can add funds to their reloadable card.
     *
     * @return bool
     */
    public function verify(): bool
    {
        return $this->hasEnoughPoints();
    }

    /**
     * Convert points to USD dollar amount.
     *
     * @param  int $points
     * @return int
     */
    public function convertPointsToDollarWithMarkup(int $points): int
    {
        $markup = $this->getMarkup() + 1;
        $amountUsd = ($points * $this->getDollarsPerPoint()) / $markup;

        return floor($amountUsd);
    }

    /**
     * Convert points to dollar amount.
     *
     * @param  int $points
     * @param  string|null $currency
     * @return int
     */
    public function convertDollarToPointsWithMarkup(int $dollar, string $currency = null): int
    {
        $exchangeToUsd = 1;
        if (!empty($currency) && strtoupper($currency) !== 'USD') {
            throw new \Exception('Currency conversion not supported yet!');
        }
        $dollarUsd = ($exchangeToUsd > 0) ? $dollar / $exchangeToUsd : $dollar;

        $markup = $this->getMarkup() + 1;
        $points = ($dollarUsd / $this->getDollarsPerPoint()) * $markup;

        return ceil($points);
    }

    /**
     * The dollar amount the participant potentially has.
     *
     * @return int The max USD amount
     */
    public function amount(): int
    {
        $userBalance = (int) str_replace(',', '', auth()->user()->points());

        return $this->convertPointsToDollarWithMarkup($userBalance);
    }

    /**
     * Get the conversion rate from points to dollar.
     *
     * @return float
     */
    public function getItem(): Item
    {
        return Item::with('catalog', 'vendor')
            ->where('sku', $this->sku())
            ->first();
    }

    /**
     * Get the conversion rate from points to dollar.
     *
     * @return float
     */
    public function getDollarsPerPoint(): float
    {
        return config('catalog.default_point_value');
    }

    /**
     * Get the conversion rate from points to dollar.
     *
     * @return float
     */
    public function getMarkup(): float
    {
        return $this->getItem()->price_markup / 100;
    }

    /**
     * Determine whether this would be the user's first reloadable order.
     *
     * @return bool
     */
    public function isFirstOrder(): bool
    {
        if ($this->isFirstOrder === null) {
            $reloadableId = Participant::query()
                ->where('user_id', auth()->id())
                ->value('reloadable_id');
            $this->isFirstOrder = !$reloadableId;
        }

        return $this->isFirstOrder;
    }

    /**
     * The date the reloadable card was created by the participant.
     *
     * @return bool|null|string
     */
    public function getReloadableCreatedAt()
    {
        if (false === $this->reloadable_created_at) {
            $participant = Participant::query()->where('user_id', auth()->id())->first();
            $this->reloadable_created_at = $participant->reloadable_created_at;
        }

        return $this->reloadable_created_at;
    }

    /**
     * Determine whether the participant has created the reloadable card.
     *
     * @return bool
     */
    public function hasCreatedReloadableCard(): bool
    {
        return ! $this->isFirstOrder();
    }

    /**
     * Determine whether the participant has not created the reloadable card
     * yet but now has enough points to do so.
     *
     * @return bool
     * @deprecated
     */
    public function hasNotCreatedCardButHasEnoughPoints(): bool
    {
        return (! $this->hasCreatedReloadableCard()) && ($this->hasEnoughPoints());
    }

    /**
     * Determine whether the participant has enough points to create the reloadable card.
     *
     * @return bool
     */
    public function hasEnoughPoints(): bool
    {
        $points = (int) str_replace([',', '.'], '', $this->cart->balance());
        $amount = $this->convertPointsToDollarWithMarkup($points);

        return ($amount >= $this->getMinRequiredAmount());
    }

    /**
     * Get the minimum required amount to create the reloadable card.
     *
     * @return int
     */
    public function getMinRequiredAmount(): int
    {
        if ($this->hasCreatedReloadableCard()) {
            $min = $this->getMinAmountAfterFirstOrder();
        } else {
            $min = $this->getMinAmountFirstOrder();
        }

        return max($min, 1);
    }

    /**
     * Get the minimum amount to create the first reloadable card.
     *
     * @return int
     */
    public function getMinAmountFirstOrder(): int
    {
        return config('catalog.vendors.hawk.reloadable.min_threshold_initial', 0);
    }

    /**
     * Get the minimum amount after the first reloadable card.
     *
     * @return int
     */
    public function getMinAmountAfterFirstOrder(): int
    {
        return config('catalog.vendors.hawk.reloadable.min_threshold_reload', 0);
    }

    /**
     * The reloadable card item sku.
     *
     * @return string
     */
    public function sku(): string
    {
        return config('catalog.vendors.hawk.reloadable.sku') ?? 'visa_reloadable';
    }
}
