<?php

namespace Ignite\Catalog\Http\Controllers;

use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Http\Requests\CreateReloadableRequest;
use Ignite\Catalog\Models\ReloadableThreshold;
use Ignite\Catalog\Repositories\OrderRepository;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReloadableController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return void
     */
    public function index()
    {
        abort(404);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @param ReloadableThreshold $threshold
     *
     * @return mixed
     */
    public function create(ReloadableThreshold $threshold)
    {
        $item = $this->itemFromAmount(1, $threshold->sku());

        return view('catalog.reloadable.create', compact('item', 'threshold'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param CreateReloadableRequest $request
     * @param OrderRepository         $orderRepository
     *
     * @return RedirectResponse
     */
    public function store(
        CreateReloadableRequest $request,
        OrderRepository $orderRepository,
        ReloadableThreshold $threshold
    ): RedirectResponse
    {
        $amount = (int) $request->get('amount');
        $minRequiredAmount = $threshold->getMinRequiredAmount();

        if ($amount < $minRequiredAmount) {
            $this->flashError('Please enter an amount greater than ' . number_format($amount, 0) . '.');
            return redirect()->back()->withInput();
        }

        if ($threshold->amount() < $amount) {
            $this->flashError(
                'Sorry, you don\'t have enough balance.'
                . ' You tried to add $' . number_format($amount, 0) . ' to your card'
                . ' but you only have $' . number_format($threshold->amount(), 0) . '.'
            );

            return redirect()->back()->withInput($request->all());
        }

        DB::beginTransaction();

        try {
            // Create a cart item from the given amount.
            // Then create the order and transaction
            $item = $this->itemFromAmount($amount, $threshold->sku());
            $orderData = $this->orderData($request, $item);
            $cartItem = $this->cartItem($item, $amount);
            $order = $orderRepository->create($orderData, collect([$cartItem]), $item->points);

            DB::commit();

            $this->flashSuccess(sprintf(
                '$%s will be added to your reloadable card.',
                number_format($amount, 0)
            ));

            return redirect()->route('catalog.checkout.show', ['number' => $order->number]);
        } catch (\Exception $e) {
            DB::rollBack();

            logger()->error($e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'code' => $e->getCode()
            ]);

            $this->flashError('Sorry, we were unable to process your order. Please contact Rewards Headquarters.');

            return redirect()->back();
        }
    }

    /**
     * Find the reloadable visa card item by sku.
     *
     * @param $amount
     * @param $sku
     * @return Item
     */
    protected function itemFromAmount(int $amount, string $sku): Item
    {
        // @todo: In possible future, we may need to consider the exchange rate
        // for foreign currency. We want the msrp/cost/price to be in USD, but
        // the actual amount to send to vendor may need to be in the foreign
        // currency conversion value.
        // We possibly may need different reloadable cards for each currency,
        // each with their own "markup" rate.

        // $amountUsd = ($item->exchange_to_usd > 0 ? $amount / $item->exchange_to_usd : $amount);
        $amountUsd = $amount;

        $item = Item::with('catalog', 'vendor')
            ->where('sku', $sku)
            ->first();

        $item->msrp = $amountUsd;
        $item->cost = $amountUsd;

        // NOTE: price_markup in the database could be 0.10
        // BUT $item->price_markup would return 10 because of Item::getPriceMarkupAttribute()
        $item->price = ($item->price_markup > 0)
            ? $amountUsd * (1 + ($item->price_markup / 100))
            : $amountUsd;
        $item->point_value = config('catalog.default_point_value');
        $item->points = ceil($item->price / $item->point_value);

        return $item;
    }

    /**
     * Create a cart item entity to pass to the repository.
     *
     * @param Item $item
     * @param int  $amount
     * @return CartItem
     */
    protected function cartItem(Item $item, int $amount): CartItem
    {
        $cartItem = new CartItem([
            'user_id' => auth()->id(),
            'catalog_id' => $item->catalog->getKey(),
            'catalog_vendor_id' => $item->vendor->getKey(),
            'item_id' => $item->getKey(),
            'quantity' => 1,
            'cost' => $item->cost,
            'price' => $item->price,
            'point_value' => config('catalog.default_point_value'),
            'points' => $item->points,
            'vendor_meta' => ['reloadable_amount' => $amount],
        ]);

        $cartItem->setRelation('item', $item);

        return $cartItem;
    }

    /**
     * Build the array of order data.
     *
     * @param Request $request
     * @param Item    $item
     * @return array
     */
    protected function orderData(Request $request, Item $item): array
    {
        $participant = Participant::query()->where('user_id', auth()->id())->first();

        return [
            'user_id' => auth()->id(),
            'quantity' => 1,
            'points' => $item->points,
            'ship_name' => $participant->fullName(),
            'ship_phone' => $request->info['ship_phone'] ?? $participant->phone1,
            'ship_email' => $request->info['ship_email'] ?? $participant->email,
            'ship_address_1' => $request->info['ship_address_1'] ?? $participant->ship_address_1,
            'ship_address_2' => $request->info['ship_address_2'] ?? $participant->address_2,
            'ship_city' => $request->info['ship_city'] ?? $participant->city,
            'ship_state' => $request->info['ship_state'] ?? strtoupper($participant->state),
            'ship_postal' => $request->info['ship_postal'] ?? $participant->postal,
            'ship_country' => 'US'
        ];
    }
}
