<?php

namespace Ignite\Catalog\Entities;

use Ignite\Catalog\Models\Product\Classifications\Card;
use Ignite\Catalog\Models\Product\Classifications\Merchandise;
use Ignite\Core\Entities\User;
use Ignite\Core\Facades\Format;
use Ignite\Packages\Presenter\Traits\Presentable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

/**
 * Class Order
 *
 * @property \Ignite\Core\Entities\User $user
 * @property \Ignite\Core\Entities\Participant $participant
 * @property \Ignite\Catalog\Collections\OrderItemCollection $items
 * @property int id
 * @property int user_id
 * @property string number
 * @property int quantity
 * @property int points
 * @property string ship_name
 * @property string ship_email
 * @property string ship_phone
 * @property string ship_address_1
 * @property string ship_address_2
 * @property string ship_address_3
 * @property string ship_city
 * @property string ship_state
 * @property string ship_postal
 * @property string ship_country
 * @property string notes
 * @property int processed
 * @property string processed_at
 * @property int cancelled
 * @property string cancelled_at
 * @property string created_at
 * @property string updated_at
 * @method static Builder processing()
 * @method static Builder processed()
 * @method static Builder cancelled()
 * @method static Builder byUser($user)
 * @method static Builder byIds($ids)
 * @method static Builder byCurrentUser()
 * @method static Builder byNumber()
 */
class Order extends Model
{
    use Presentable;

    /** @var string */
    protected $presenter = \Ignite\Catalog\Presenters\OrderPresenter::class;

    /** @var string */
    protected $primaryKey = 'id';

    /** @var string */
    protected $table = 'catalog_order';

    /** @var array */
    protected $guarded = [];

    /**
     * The relationship to the user.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(\Ignite\Core\Entities\User::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the participant.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo(\Ignite\Core\Entities\Participant::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the catalog order line item.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(\Ignite\Catalog\Entities\OrderItem::class, 'catalog_order_id', 'id');
    }

    /**
     * Calculate the total points for the order.
     *
     * @return string
     */
    public function totalPoints()
    {
        return Format::amount($this->items->reduce(function ($previous, OrderItem $orderItem) {
            return $previous + ($orderItem->points * $orderItem->quantity);
        }, 0));
    }

    /**
     * Calculate the total quantity of items in the order.
     *
     * @return string
     */
    public function totalQuantity()
    {
        return number_format($this->items->sum('quantity'), 0);
    }

    /**
     * Determine if the order is still processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return (! $this->processed && ! $this->cancelled);
    }

    /**
     * Determine if the order can be cancelled.
     *
     * @return bool
     */
    public function isCancelable()
    {
        return ! $this->isProcessed() && ! $this->isCancelled();
    }

    /**
     * Determine if the order is cancelled.
     *
     * @return int
     */
    public function isCancelled()
    {
        return (int) $this->cancelled;
    }

    /**
     * Determine if the order can be processed.
     *
     * @return bool
     */
    public function isProcessable()
    {
        return ! $this->isProcessed() && ! $this->isCancelled();
    }

    /**
     * Determine if the order is processed.
     *
     * @return int
     */
    public function isProcessed()
    {
        return (int) $this->processed;
    }

    /**
     * Determine whether the cart has at least one physical item.
     *
     * @return bool
     */
    public function hasPhysical()
    {
        return $this->countPhysical() > 0;
    }

    /**
     * Determine whether the cart contains only physical items.
     *
     * @return bool
     */
    public function hasOnlyPhysical()
    {
        return $this->items->count() === $this->countPhysical();
    }

    /**
     * The number of physical items in the cart.
     *
     * @return int
     */
    public function countPhysical()
    {
        return $this->items->filter(function (OrderItem $model) {
            return $model->item->getClassInstance()->isPhysical();
        })->count();
    }

    /**
     * Determine whether the cart has at least one non-physical item.
     *
     * @return bool
     */
    public function hasNonPhysical()
    {
        return $this->countNonPhysical() > 0;
    }

    /**
     * Determine whether the cart contains only non-physical items.
     *
     * @return bool
     */
    public function hasOnlyNonPhysical()
    {
        return $this->items->count() === $this->countNonPhysical();
    }

    /**
     * The number of non-physical items in the cart.
     *
     * @return int
     */
    public function countNonPhysical()
    {
        return $this->items->filter(function (OrderItem $model) {
            return ! $model->item->getClassInstance()->isPhysical();
        })->count();
    }

    /**
     * Determine whether the cart has at least one merchandise item.
     *
     * @return bool
     */
    public function hasMerchandise()
    {
        return $this->countMerchandise() > 0;
    }

    /**
     * Determine whether the cart contains only merchandise items.
     *
     * @return bool
     */
    public function hasOnlyMerchandise()
    {
        return $this->items->count() === $this->countMerchandise();
    }

    /**
     * The number of merchandise items in the cart.
     *
     * @return int
     */
    public function countMerchandise()
    {
        return $this->items->filter(function (OrderItem $model) {
            return ($model->item->getClassInstance() instanceof Merchandise);
        })->count();
    }

    /**
     * Determine whether the cart has at least one card type item.
     *
     * @return bool
     */
    public function hasCardType()
    {
        return $this->countCardType() > 0;
    }

    /**
     * Determine whether the cart contains only card type items.
     *
     * @return bool
     */
    public function hasOnlyCardType()
    {
        return $this->items->count() === $this->countCardType();
    }

    /**
     * The number of card type items in the cart.
     *
     * @return int
     */
    public function countCardType()
    {
        return $this->items->filter(function (OrderItem $model) {
            return ($model->item->getClassInstance() instanceof Card);
        })->count();
    }

    /**
     * Scope a query by attaching a user.
     *
     * @param  Builder  $query
     * @param  User|int $user
     * @return Builder
     */
    public function scopeByUser(Builder $query, $user)
    {
        $user = ($user instanceof User) ? $user->getKey() : $user;
        $query->where('user_id', $user);

        return $query;
    }

    /**
     * Query for multiple IDs.
     *
     * @param  Builder      $query
     * @param  array|string $ids
     * @return Builder
     */
    public function scopeByIds(Builder $query, $ids)
    {
        if (is_string($ids)) {
            $ids = preg_split('/\s?,\s?/', $ids);
        }

        $query->whereIn('id', $ids);

        return $query;
    }

    /**
     * Scope a query by attaching the current user.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeByCurrentUser(Builder $query)
    {
        $query->where('user_id', auth()->user()->getKey());

        return $query;
    }

    /**
     * Scope a query by order number.
     *
     * @param  Builder $query
     * @param  string  $number
     * @return Builder
     */
    public function scopeByNumber(Builder $query, $number)
    {
        $query->where('number', $number);

        return $query;
    }

    /**
     * Scope a query by processed only.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeProcessed(Builder $query)
    {
        $query->where('processed', 1);

        return $query;
    }

    /**
     * Scope a query by processing only.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeProcessing(Builder $query)
    {
        $query->where('processed', 0)->where('cancelled', 0);

        return $query;
    }

    /**
     * Scope a query by cancelled only.
     *
     * @param  Builder $query
     * @return Builder
     */
    public function scopeCancelled(Builder $query)
    {
        $query->where('cancelled', 1);

        return $query;
    }

    /**
     * Determine whether all items have been processed.
     *
     * @return bool
     */
    public function hasProcessedAllItems()
    {
        return $this->items->processed()->count() === $this->items->count();
    }

    /**
     * Determine whether all items have been cancelled.
     *
     * @return bool
     */
    public function hasCancelledAllItems()
    {
        return $this->items->cancelled()->count() === $this->items->count();
    }
}
