<?php

namespace Ignite\Catalog\Repositories;

use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Ignite\Catalog\Contracts\CatalogRepository as CatalogRepositoryContract;
use Ignite\Catalog\Entities\Catalog;
use Ignite\Catalog\Entities\Item;
use Ignite\Core\Entities\Participant;

class CatalogRepository implements CatalogRepositoryContract
{
    /**
     * Find all active catalogs.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return Catalog::query();
    }

    /**
     * Find all catalogs.
     *
     * @return \Illuminate\Support\Collection
     */
    public function findAll()
    {
        return $this->query()->get();
    }

    /**
     * Find all active catalogs.
     *
     * @return \Illuminate\Support\Collection
     */
    public function findAllActive()
    {
        return $this->query()->onlyActive()->get();
    }

    /**
     * Find all active catalogs.
     *
     * @param  Participant $participant
     * @return \Illuminate\Support\Collection
     */
    public function findAllActiveForParticipant(Participant $participant)
    {
        return $this->query()->onlyActive()->byParticipant($participant)->get();
    }

    /**
     * Find a catalog.
     *
     * @param  mixed  $id
     * @param  string $column
     * @return \Illuminate\Database\Eloquent\Model|Catalog
     */
    public function find($id, $column = 'id')
    {
        return $this->query()->where($column, $id)->first();
    }

    /**
     * Create a catalog.
     *
     * @param  Collection  $data
     * @return Catalog
     */
    public function create(Collection $data)
    {
        $catalog = Catalog::create($data->toArray());

        return $catalog;
    }

    /**
     * Update a catalog.
     *
     * @param  int  $id
     * @param  Collection  $data
     * @return Catalog
     */
    public function update($id, Collection $data)
    {
        $catalog = $this->find($id);

        $catalog->update($data->toArray());

        return $catalog;
    }

    /**
     * Find a catalog by code.
     *
     * @param  string $code
     * @return \Illuminate\Database\Eloquent\Model|Catalog
     */
    public function findByCode($code)
    {
        return $this->find($code, 'code');
    }

    /**
     * Search the catalog.
     *
     * @param  Catalog     $catalog
     * @param  string|null $keyword
     * @param  string|null $category
     * @param  string      $sort
     * @param  int         $perPage
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function search($catalog, $keyword = null, $category = null, $sort = 'name_asc', $perPage = 24)
    {
        /** @var Item $builder */
        $builder = Item::with('catalog', 'vendor')
            ->leftJoin('catalog_favorite', function (JoinClause $join) {
                $join->on('catalog_favorite.item_id', '=', 'catalog_item.id');
                $join->on('catalog_favorite.user_id', '=', DB::raw(auth()->user()->getKey()));
            })
            ->byCatalog($catalog)
            ->onlyActive()
            ->onlyVendorActive()
            ->onlyVisible()
            ->addSelect('catalog_item.*', 'catalog_favorite.id as favorite');

        // Scope the keyword
        if ($keyword) {
            $builder->matchAgainst($keyword);
        }

        // Scope the category
        if ($category) {
            $builder->byCategory($category);
        }

        // Scope the order
        if ($sort) {
            $scope = camel_case('order_by_' . $sort);
            try {
                $builder->$scope();
            } catch (\Exception $e) {
                $builder->orderByNameAsc();
            }
        }

        return $builder->paginate($perPage);
    }

    /**
     * Paginate the catalogs.
     *
     * @param  int $perPage
     * @param  bool $activeOnly
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function paginate($perPage = 25, $activeOnly = true)
    {
        $builder = $this->query();

        if ($activeOnly) {
            $builder->activeOnly();
        }

        return $builder->paginate($perPage);
    }

    /**
     * Delete a catalog and any associated data.
     *
     * @param  int  $id
     * @return Catalog
     */
    public function delete(int $id)
    {
        $catalog = $this->find($id);

        $catalog->forceDelete();

        return $catalog;
    }

    /**
     * Delete the records with the given IDs.
     *
     * @param  array $ids
     * @return bool
     */
    public function deleteMany(array $ids)
    {
        return app('db')->transaction(function () use ($ids) {
            return Catalog::query()->whereIn('id', $ids)->delete();
        });
    }
}
